package com.atlassian.bitbucket.request;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * Value that is resolved in the current {@link RequestContext}, if one is active. Conceptually similar to a
 * {@link ThreadLocal}, but tied to the current {@link RequestContext} instead of the current thread.
 * <p>
 * Note that the request context can be propagated to background threads such as event listener threads and repository
 * hook handler threads. As a result, the object stored in a {@code RequestLocal} could be accessed from different
 * threads and care should be taken to ensure the stored object is thread safe when interacting with it from multiple
 * threads (e.g. the request thread, an asynchronous event listener and/or a repository hook).
 *
 * @param <T> the value type
 * @since 7.9
 */
public interface RequestLocal<T> {

    /**
     * Attempts to compute the value given the current value. The value returned by the {@code mappingFunction} is
     * stored as the value for current {@link RequestContext}.
     * <p>
     * If there is no current {@link RequestContext}, the {@code mappingFunction} is still invoked but the computed
     * value is not stored.
     *
     * @param mappingFunction function that computes the value given the current value which may be {@code null} if
     *                        no value is present.
     * @return the computed value
     */
    T compute(@Nonnull Function<? super T, ? extends T> mappingFunction);

    /**
     * Returns the value, if present. If an {@link RequestContext} is active, but no value is set, the value supplied
     * by the {@code missingValueSupplier} is stored and returned.
     * <p>
     * If no {@link RequestContext} is active, {@code missingValueSupplier} is still invoked, but the computed value
     * is not stored.
     *
     * @param missingValueSupplier supplier of the value when
     * @return the value, possibly supplied by {@code missingValueSupplier}
     */
    T computeIfAbsent(@Nonnull Supplier<? extends T> missingValueSupplier);

    /**
     * @return the value, or {@code null} if not set or no {@link RequestContext} is active
     */
    @Nullable
    T get();

    /**
     * If a value is present, invoke the specified consumer with the value, otherwise do nothing.
     *
     * @param consumer block to be executed if a value is present
     */
    void ifPresent(@Nonnull Consumer<? super T> consumer);

    /**
     * @return {@code true} is a {@link RequestContext} is currently active, otherwise {@code false}
     */
    boolean isActive();

    /**
     * @return {@code true} when a value other than {@code null} is defined in the current {@link RequestContext}.
     */
    boolean isPresent();

    /**
     * Removes the value from the current {@link RequestContext}, if present.
     *
     * @return the removed value, if the value was removed; otherwise {@code null}
     */
    @Nullable
    T remove();

    /**
     * Removes the value from the current {@link RequestContext}, if a non-null value is present and the
     * {@code predicate} matches the value. If no value is present, the predicate will not be called. The Predicate
     * will never be called with a {@code null} value.
     *
     * @param predicate the predicate
     * @return the removed value, if the value was removed; otherwise {@code null}
     */
    @Nullable
    T removeIf(@Nonnull Predicate<? super T> predicate);

    /**
     * Sets the value. Setting a {@code null} value is equivalent to calling {@link #remove()}.
     *
     * @param value the value
     * @return {@code true} if the value was set, or {@code false} if no {@link RequestContext} is active
     */
    boolean set(@Nullable T value);
}
