package com.atlassian.bitbucket.repository;

import javax.annotation.Nonnull;
import java.util.Objects;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Objects.requireNonNull;

/**
 * A simple, immutable implementation of {@link RefChange}. To enhance compatibility across releases, instances must
 * be created using the nested {@link Builder builder}.
 */
public class SimpleRefChange extends SimpleMinimalRefChange implements RefChange {

    private final String fromHash;
    private final String toHash;

    /**
     * @param builder builder to construct the ref change from
     * @since 5.10
     */
    protected SimpleRefChange(@Nonnull AbstractBuilder<?> builder) {
        super(builder);

        this.fromHash = requireNonNull(builder.fromHash, "fromHash");
        this.toHash = requireNonNull(builder.toHash, "toHash");
    }

    @Nonnull
    @Override
    public String getFromHash() {
        return fromHash;
    }

    @Nonnull
    @Override
    public MinimalRef getRef() {
        return super.getRef();
    }

    @Nonnull
    @Override
    public String getToHash() {
        return toHash;
    }

    @Nonnull
    @Override
    public RefChangeType getType() {
        return super.getType();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        SimpleRefChange that = (SimpleRefChange) o;
        return fromHash.equals(that.fromHash) && toHash.equals(that.toHash);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), fromHash, toHash);
    }

    @Override
    public String toString() {
        return getRef().getId() + ": " + getFromHash() + " -> " + getToHash() + " (" + getType() + ")";
    }

    public static class Builder extends AbstractBuilder<SimpleRefChange.Builder> {

        public Builder() {
        }

        public Builder(@Nonnull RefChange refChange) {
            super(refChange);
        }

        @Nonnull
        public SimpleRefChange build() {
            validate();

            return new SimpleRefChange(self());
        }

        @Nonnull
        public Builder from(@Nonnull Ref value) {
            return super.from(value);
        }

        @Nonnull
        public Builder fromHash(@Nonnull String value) {
            return super.fromHash(value);
        }

        @Nonnull
        public Builder ref(@Nonnull MinimalRef value) {
            return super.ref(value);
        }

        @Override
        public Builder self() {
            return this;
        }

        @Nonnull
        public Builder to(@Nonnull Ref value) {
            return super.to(value);
        }

        @Nonnull
        public Builder toHash(@Nonnull String value) {
            return super.toHash(value);
        }

        @Nonnull
        public Builder type(@Nonnull RefChangeType value) {
            return super.type(value);
        }
    }

    /**
     * @since 5.10
     */
    public abstract static class AbstractBuilder<B extends AbstractBuilder<B>> extends SimpleMinimalRefChange.AbstractMinimalRefChangeBuilder<B, RefChange> {

        private String fromHash;
        private String toHash;

        protected AbstractBuilder() {
        }

        public AbstractBuilder(@Nonnull RefChange change) {
            super(change);

            fromHash = requireNonNull(change, "change").getFromHash();
            toHash = change.getToHash();
        }

        @Nonnull
        public B from(@Nonnull Ref value) {
            fromHash = requireNonNull(value, "ref").getLatestCommit();

            return ref(value);
        }

        @Nonnull
        public B fromHash(@Nonnull String value) {
            fromHash = requireNonNull(value, "value");

            return self();
        }

        @Nonnull
        @Override
        public B ref(@Nonnull MinimalRef value) {
            return super.ref(value);
        }

        @Nonnull
        public B to(@Nonnull Ref value) {
            requireNonNull(value, "ref");
            if (value instanceof Tag) {
                Tag tag = (Tag) value;
                // If the tag is an annotated tag, then the toHash should be the annotated tag object rather than
                // the tagged commit.
                toHash = firstNonNull(tag.getHash(), tag.getLatestCommit());
            } else {
                toHash = value.getLatestCommit();
            }

            return ref(value);
        }

        @Nonnull
        public B toHash(@Nonnull String value) {
            toHash = requireNonNull(value, "value");

            return self();
        }

        @Nonnull
        @Override
        public B type(@Nonnull RefChangeType value) {
            return super.type(value);
        }

        /**
         * @deprecated in 7.11 for removal in 8.0. Validation has been moved to the constructor of {@link SimpleRefChange}.
         */
        @Deprecated
        protected void validate() {
            // Validation has been moved to the constructor
        }

        protected abstract B self();
    }
}
