package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * Request for searching for pull requests by participant. The purpose and behaviour of each field is described on its
 * accessor. Used to add searching of specific participants to a {@link PullRequestSearchRequest}.
 */
public class PullRequestParticipantRequest {

    private final PullRequestRole role;
    private final Set<PullRequestParticipantStatus> statuses;
    private final String username;

    private PullRequestParticipantRequest(Builder builder) {
        statuses = ImmutableSet.copyOf(builder.statuses);
        role = builder.role;
        username = builder.username;
    }

    /**
     * When set, limits the pull requests to search for by a specified {@link PullRequestRole pull request role}
     * for the {@link ApplicationUser user}.
     *
     * @return the set {@link PullRequestRole pull request role}
     */
    @Nullable
    public PullRequestRole getRole() {
        return role;
    }

    /**
     * Limits the pull requests to search for by the participant {@link PullRequestParticipantStatus status}
     * @return the added {@link PullRequestParticipantStatus statuses}
     * @since 4.2
     */
    @Nonnull
    public Set<PullRequestParticipantStatus> getStatuses() {
        return statuses;
    }

    /**
     * Restricts the search to a specified {@link ApplicationUser user} with the given username.
     *
     * @return the username for the user to search for
     */
    @Nonnull
    public String getUsername() {
        return username;
    }

    public static class Builder extends BuilderSupport {

        private final EnumSet<PullRequestParticipantStatus> statuses;
        private final String username;

        private PullRequestRole role;

        public Builder(@Nonnull String username) {
            this.username = requireNonNull(username, "username");
            statuses = EnumSet.noneOf(PullRequestParticipantStatus.class);
        }

        /**
         * Assembles a new {@link PullRequestParticipantRequest} from the provided values.
         *
         * @return a new pull request participant search request instance
         */
        @Nonnull
        public PullRequestParticipantRequest build() {
            return new PullRequestParticipantRequest(this);
        }

        /**
         * Limits the pull requests to search for by a specified {@link PullRequestRole pull request role} for the
         * {@link ApplicationUser user}.
         *
         * @param value the {@link PullRequestRole pull request role}
         * @return the current builder
         */
        @Nonnull
        public Builder role(@Nullable PullRequestRole value) {
            role = value;
            return this;
        }

        /**
         * Limits the pull requests to search for by the participant status
         * @param value the status to search for
         * @return the current builder
         * @since 4.2
         */
        @Nonnull
        public Builder status(@Nullable PullRequestParticipantStatus value) {
            addIf(Objects::nonNull, statuses, value);
            return this;
        }

        /**
         * Limits the pull requests to search for by the participant status
         * @param values the statuses to search for
         * @return the current builder
         * @since 4.2
         */
        @Nonnull
        public Builder statuses(@Nonnull Iterable<PullRequestParticipantStatus> values) {
            addIf(Objects::nonNull, statuses, values);
            return this;
        }
    }
}
