package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.job.Job;
import com.atlassian.bitbucket.job.JobMessage;
import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * @since 5.13
 */
public interface MigrationService {

    /**
     * Cancels the export job. This happens immediately and the resulting archive is not usable.
     *
     * @param jobId the ID of the export job
     * @return export job or {@link Optional#empty()} if the export job doesn't exist
     * @throws IllegalStateException if the job has been canceled already, or if it had ended
     * @see Job
     */
    @Nonnull
    Optional<Job> cancelExport(long jobId);

    /**
     * Cancels the import job. The currently importing repository will finish importing and then the import will be
     * canceled.
     *
     * @param jobId the ID of the import job
     * @return import job or {@link Optional#empty()} if the import job doesn't exist
     * @throws IllegalStateException if the job has been canceled already, or if it had ended
     * @see Job
     */
    @Nonnull
    Optional<Job> cancelImport(long jobId);

    /**
     * Retrieves the export job.
     *
     * @param jobId the ID of the export job
     * @return export job
     * @see Job
     */
    @Nonnull
    Optional<Job> getExportJob(long jobId);

    /**
     * Retrieves the import job.
     *
     * @param jobId the ID of the import job
     * @return import job
     * @see Job
     */
    @Nonnull
    Optional<Job> getImportJob(long jobId);

    /**
     * Resolves export items without triggering an export.
     *
     * @param request a request object describing the repositories to be exported
     * @return the items to be exported if an export job with the given request were triggered
     */
    @Nonnull
    Stream<Scope> previewExport(@Nonnull ExportRequest request);

    /**
     * Finds all {@link JobMessage job messages} matching the supplied request object for an export job.
     *
     * @param request     the request
     * @param pageRequest the page request
     * @return the page of matching {@link JobMessage job messages}
     * @throws IllegalArgumentException if the {@link MigrationJobMessageSearchRequest#getJobId() job ID} on the request
     *                                  is not an export job ID
     */
    @Nonnull
    Page<JobMessage> searchExportJobMessages(@Nonnull MigrationJobMessageSearchRequest request,
                                             @Nonnull PageRequest pageRequest);

    /**
     * Finds all {@link JobMessage job messages} matching the supplied request object for an import job.
     *
     * @param request     the request
     * @param pageRequest the page request
     * @return the page of matching {@link JobMessage job messages}
     * @throws IllegalArgumentException if the {@link MigrationJobMessageSearchRequest#getJobId() job ID} on the request
     *                                  is not an import job ID
     */
    @Nonnull
    Page<JobMessage> searchImportJobMessages(@Nonnull MigrationJobMessageSearchRequest request,
                                             @Nonnull PageRequest pageRequest);

    /**
     * Export the repositories specified in the {@link ExportRequest export request}.
     *
     * @param request a request object describing the repositories to be exported
     * @return the export job, the export job contains an export ID which could be used to retrieve the job status
     * @throws MaxConcurrentMigrationJobsException if the maximum number of concurrent migration jobs are already
     *                                             running on this node
     * @see ExportRequest
     */
    @Nonnull
    Job startExport(@Nonnull ExportRequest request);

    /**
     * Import the data contained in the archive specified in the {@link ImportRequest import request}.
     *
     * @param request a request object specifying the archive to import
     * @return the import job, the import job contains an import ID which could be used to retrieve the job status
     * @throws MaxConcurrentMigrationJobsException if the maximum number of concurrent migration jobs are already
     *                                             running on this node
     * @see ImportRequest
     */
    @Nonnull
    Job startImport(@Nonnull ImportRequest request);
}
