package com.atlassian.bitbucket.hook.script;

import javax.annotation.Nonnull;

/**
 * Denotes a {@link HookScript}'s type.
 * <p>
 * Because {@link #PRE pre}-style hook scripts are able to <i>reject</i> changes, they are generally written with
 * different expectations/behaviors than {@link #POST post}-style scripts, which can only <i>react</i> to changes.
 * Each {@link HookScript} has a specific type, and is invoked accordingly.
 *
 * @since 6.2
 */
public enum HookScriptType {

    /**
     * Post-style hooks are invoked after changes have been accepted. ({@code PostRepositoryHook})
     *
     * @see com.atlassian.bitbucket.hook.repository.RepositoryHookService#postUpdate
     */
    POST(2),
    /**
     * Pre-style hooks are invoked before changes are applied to a repository to review, and potentially reject,
     * the changes. ({@code PreRepositoryHook})
     *
     * @see com.atlassian.bitbucket.hook.repository.RepositoryHookService#preUpdate
     */
    PRE(1);

    private final int id;

    HookScriptType(int id) {
        this.id = id;
    }

    /**
     * @param id the ID to retrieve a type for
     * @return the type associated with the ID
     * @throws IllegalArgumentException if no type has the specified ID
     */
    @Nonnull
    public static HookScriptType fromId(int id) {
        for (HookScriptType value : values()) {
            if (value.getId() == id) {
                return value;
            }
        }

        throw new IllegalArgumentException("No HookScriptType is associated with ID [" + id + "]");
    }

    /**
     * @return the type's ID
     */
    public int getId() {
        return id;
    }
}
