package com.atlassian.bitbucket.event.repository;

import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.RepositoryService;
import com.atlassian.bitbucket.util.CancelState;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Raised just before a {@link Repository repository} is deleted. This event is synchronous, allowing listeners to
 * perform clean up in the same database transaction where the repository will be deleted.
 * <p>
 * This event is {@link CancelableEvent cancelable}. A listener may prevent the repository from being deleted by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i>
 * prevent the repository from being deleted; the exception will be logged and ignored.
 *
 * @see RepositoryService#delete(Repository)
 */
public class RepositoryDeletionRequestedEvent extends RepositoryEvent implements CancelableEvent {

    private final CancelState cancelState;
    private final Iterable<Integer> forkIds;

    /**
     * @param source      the object on which the event initially occurred
     * @param repository  repository that was deleted
     * @param cancelState the cancel state
     * @param forkIds     IDs of forks
     * @since 5.6
     */
    public RepositoryDeletionRequestedEvent(@Nonnull Object source, @Nonnull Repository repository,
                                            @Nonnull CancelState cancelState, @Nonnull Iterable<Integer> forkIds) {
        super(source, repository);

        this.cancelState = requireNonNull(cancelState, "cancelState");
        this.forkIds = ImmutableSet.copyOf(requireNonNull(forkIds, "forkIds"));
    }

    /**
     * Cancels repository deletion, providing a message explaining why.
     *
     * @param message a descriptive message explaining why the operation has been canceled
     */
    @Override
    public void cancel(@Nonnull KeyedMessage message) {
        cancelState.cancel(message);
    }

    /**
     * Retrieves a flag indicating whether repository deletion has already been canceled by another listener.
     *
     * @return {@code true} if another listener has already canceled repository deletion; otherwise, {@code false}
     */
    @Override
    public boolean isCanceled() {
        return cancelState.isCanceled();
    }

    /**
     * @return the IDs of all repositories that are forks of the repository being deleted
     * @since 5.6
     */
    @Nonnull
    public Iterable<Integer> getForkIds() {
        return forkIds;
    }
}
