package com.atlassian.bitbucket.event.content;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.bitbucket.event.branch.BranchChangedEvent;
import com.atlassian.bitbucket.event.repository.RepositoryRefsChangedEvent;
import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import java.io.InputStream;

import static java.util.Objects.requireNonNull;

/**
 * An event that is raised after a file is edited on the server.
 * <p>
 * Listeners that are interested in all repository changes should listen for
 * {@link RepositoryRefsChangedEvent}s instead.
 *
 * @since 4.13
 */
@AsynchronousPreferred
@EventName("stash.content.file.edited")
public class FileEditedEvent extends BranchChangedEvent {

    private final InputSupplier<InputStream> content;
    private final String message;
    private final String path;

    /**
     * @param source        the object on which the event initially occurred
     * @param repository    the repository the file is in
     * @param branch        the branch the file is on
     * @param refChange     the ref change
     * @param path          path to the file
     * @param content       the new content of the file
     * @param message       the commit message
     * @since 5.0
     */
    public FileEditedEvent(@Nonnull Object source, @Nonnull Repository repository, @Nonnull Branch branch,
                           @Nonnull RefChange refChange, @Nonnull String path,
                           @Nonnull InputSupplier<InputStream> content, @Nonnull String message) {
        super(source, repository, branch, refChange);

        this.path = requireNonNull(path, "path");
        this.content = requireNonNull(content, "content");
        this.message = requireNonNull(message, "message");
    }

    /**
     * @return the new content of the edited file
     * @since 5.0
     */
    @Nonnull
    public InputSupplier<InputStream> getContent() {
        return content;
    }

    /**
     * @return the commit message
     * @since 5.0
     */
    @Nonnull
    public String getMessage() {
        return message;
    }

    /**
     * @return the path of the edited (or created) file
     */
    @Nonnull
    public String getPath() {
        return path;
    }
}
