package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

public class DiffContext {

    private final int contextLines;
    private final String fromHash;
    private final int maxLineLength;
    private final int maxLines;
    private final String toHash;
    private final DiffWhitespace whitespace;

    private DiffContext(Builder builder) {
        contextLines = builder.contextLines;
        fromHash = builder.fromHash;
        maxLineLength = builder.maxLineLength;
        maxLines = builder.maxLines;
        toHash = builder.toHash;
        whitespace = builder.whitespace;
    }

    public int getContextLines() {
        return contextLines;
    }

    @Nullable
    public String getFromHash() {
        return fromHash;
    }

    public int getMaxLineLength() {
        return maxLineLength;
    }

    public int getMaxLines() {
        return maxLines;
    }

    @Nonnull
    public String getToHash() {
        return toHash;
    }

    @Nonnull
    public DiffWhitespace getWhitespace() {
        return whitespace;
    }

    public static class Builder extends BuilderSupport {

        private final String fromHash;
        private final String toHash;

        private int contextLines;
        private int maxLineLength;
        private int maxLines;
        private DiffWhitespace whitespace;

        public Builder(@Nullable String fromHash, @Nonnull String toHash) {
            this.fromHash = StringUtils.trimToNull(fromHash);
            this.toHash = checkNotBlank(toHash, "toHash");

            whitespace = DiffWhitespace.SHOW;
        }

        @Nonnull
        public DiffContext build() {
            return new DiffContext(this);
        }

        @Nonnull
        public Builder contextLines(int value) {
            contextLines = value;

            return this;
        }

        @Nonnull
        public Builder maxLineLength(int value) {
            maxLineLength = value;

            return this;
        }

        @Nonnull
        public Builder maxLines(int value) {
            maxLines = value;

            return this;
        }

        @Nonnull
        public Builder whitespace(@Nonnull DiffWhitespace value) {
            whitespace = requireNonNull(value, "whitespace");

            return this;
        }
    }
}
