package com.atlassian.bitbucket.util;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.mutable.MutableLong;

import javax.annotation.Nonnull;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.nio.file.attribute.*;
import java.time.Instant;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import static com.atlassian.bitbucket.util.MoreCollectors.toImmutableSet;
import static java.nio.file.attribute.AclEntryPermission.*;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;

/**
 * Additional utility methods missing from {@link Files}.
 *
 * @since 5.11
 */
public class MoreFiles {

    private static final Set<AclEntryPermission> EXECUTE_ACL_PERMISSIONS = ImmutableSet.of(EXECUTE);
    private static final Set<AclEntryPermission> READ_ACL_PERMISSIONS = ImmutableSet.of(READ_ACL, READ_ATTRIBUTES,
            READ_DATA, READ_NAMED_ATTRS);
    private static final Set<AclEntryPermission> WRITE_ACL_PERMISSIONS = ImmutableSet.of(WRITE_ACL, WRITE_ATTRIBUTES,
            WRITE_DATA, WRITE_NAMED_ATTRS, WRITE_OWNER, APPEND_DATA, DELETE, DELETE_CHILD, SYNCHRONIZE);

    private MoreFiles() {
        throw new UnsupportedOperationException(getClass().getName() +
                " is a utility class and should not be instantiated");
    }

    /**
     * Deletes any files or subdirectories in the specified directory and leaves the directory empty.
     *
     * @param directory the directory to clean
     * @throws IOException if the path does not denote a directory, or the directory's contents could not
     *                     be deleted
     */
    public static void cleanDirectory(@Nonnull Path directory) throws IOException {
        requireNonNull(directory, "directory");

        try (Stream<Path> entries = Files.list(directory)) {
            for (Path entry : (Iterable<Path>) entries::iterator) {
                BasicFileAttributes attributes = Files.readAttributes(entry, BasicFileAttributes.class);
                if (attributes.isDirectory()) {
                    //If the entry is a directory, recursively delete it
                    deleteRecursively(entry);
                } else {
                    //Otherwise, for files and symbolic links, delete them directly
                    Files.delete(entry);
                }
            }
        }
    }

    /**
     * Registers the provided path to be deleted when the JVM exits.
     * <p>
     * Delete-on-exit is performed on a <i>best-effort</i> basis, and should not be relied upon as the primary
     * solution for deleting files. Additionally, it only works for files and <i>empty</i> directories.
     *
     * @param path the path to register for deletion when the JVM exits
     * @see File#deleteOnExit()
     */
    public static void deleteOnExit(@Nonnull Path path) {
        requireNonNull(path, "path").toFile().deleteOnExit();
    }

    /**
     * {@link #deleteRecursively Recursively deletes} the specified path, suppressing any {@link IOException}s
     * that are thrown.
     *
     * @param path the path to delete, which may be a file or a directory
     * @return {@code true} if the path was deleted; otherwise, {@code false} if it did not exist or could not
     *         be deleted
     */
    public static boolean deleteQuietly(@Nonnull Path path) {
        try {
            deleteRecursively(path);
        } catch (IOException ignored) {
            return false;
        }

        return true;
    }

    /**
     * Recursively deletes the specified path.
     * <p>
     * If the specified {@link Path} denotes a directory, the directory's contents are <i>recursively</i>
     * deleted, depth-first, to empty the directory so it can be deleted. If any files or subdirectories
     * can't be deleted, an exception will be thrown. <i>Note that some files and subdirectories may have
     * been deleted prior to the exception being thrown.</i> Additionally, if the directory contains any
     * symbolic links the <i>links themselves</i> are deleted, not their targets.
     * <p>
     * If the specified {@link Path} denotes a symbolic link, the <i>symbolic link itself</i> is deleted,
     * rather than the target of the link. If the path is a symbolic link to a directory, that directory's
     * contents <i>are not removed</i>.
     * <p>
     * If the specified {@link Path} denotes a file, it is deleted.
     * <p>
     * If the specified path does not exist, nothing happens and no exception is thrown.
     *
     * @param path the path to delete, which may be a file or a directory
     * @throws IOException if the specified path cannot be deleted
     */
    public static void deleteRecursively(@Nonnull Path path) throws IOException {
        requireNonNull(path, "path");

        //If "path" is a file, visitFile will be called for it. Otherwise, if it's a directory, it will
        //be recursively traversed, depth-first
        Files.walkFileTree(path, new SimpleFileVisitor<Path>() {

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                //By postVisitDirectory, any files in the directory should have been deleted
                return deleteIfExists(dir);
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                return deleteIfExists(file);
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                if (exc instanceof FileNotFoundException || exc instanceof NoSuchFileException) {
                    //If we try to delete a file and it's already been deleted, move on
                    return FileVisitResult.CONTINUE;
                }
                throw exc;
            }

            //Why not Files.deleteIfExists? Because checking existence and deleting aren't atomic, so we
            //need to handle the same exceptions anyway
            private FileVisitResult deleteIfExists(Path path) throws IOException {
                try {
                    Files.delete(path);
                } catch (FileNotFoundException | NoSuchFileException ignored) {
                    //Ignore attempts to delete files that don't exist. This is done instead of calling
                    //Files.deleteIfExists because checking existence and then performing the delete is
                    //not atomic, so we'd need to handle these exceptions anyway
                }

                return FileVisitResult.CONTINUE;
            }
        });
    }

    /**
     * Gets the {@link Files#getLastModifiedTime last modified time} for the specified {@link Path}, in
     * milliseconds.
     * <p>
     * Like {@link File#lastModified}, if the path does not exist or its attributes cannot be read, {@code 0L}
     * is returned rather than throwing an exception. Use {@link Files#getLastModifiedTime} directly if an
     * exception is desired.
     *
     * @param path the path to retrieve the last modified time for
     * @return the file's last modified time, in milliseconds, or {@code 0L} if the time could not be determined
     */
    public static long getLastModified(@Nonnull Path path) {
        requireNonNull(path, "path");

        try {
            return Files.getLastModifiedTime(path).toMillis();
        } catch (IOException ignored) {
            return 0L;
        }
    }

    /**
     * Returns {@code true} if the specified path is contained within the {@code expectedParent}. This should be
     * used to ensure paths created with user-entered data don't "escape" the specified parent, to prevent path
     * traversal attacks.
     *
     * @param path           the path to validate
     * @param expectedParent the required parent directory
     * @return {@code true} if {@code path} is contained within {@code expectedParent}; otherwise, {@code false}
     * @throws IllegalArgumentException if {@code expectedParent} does not exist or is not a directory
     * @throws IOException if the real path for either of the provided paths cannot be resolved
     */
    public static boolean isWithin(@Nonnull Path path, @Nonnull Path expectedParent) throws IOException {
        requireNonNull(path, "path");
        requireNonNull(expectedParent, "expectedParent");
        if (!Files.isDirectory(expectedParent)) {
            throw new IllegalArgumentException("expectedParent is not a directory");
        }

        //File.getCanonical*() doesn't throw if the file doesn't exist, but Path.toRealPath() does. If that
        //happens, we fall back on using File.getCanonical
        Path test;
        try {
            test = path.toRealPath();
        } catch (NoSuchFileException e) {
            //There's no call, or set of calls, on Path that's identical to what File.getCanonicalPath() is
            //doing. Rather than try to emulate the behavior, it's safer to switch to File and back again.
            //
            //An example of where using, for example, Path.normalize() to try and emulate getCanonicalPath()
            //fails is using short names on Windows (e.g. PROGRA~1). normalize() does not "expand" such paths,
            //but toRealPath() and getCanonicalPath() do
            test = Paths.get(path.toFile().getCanonicalPath());
        }

        return test.startsWith(expectedParent.toRealPath());
    }

    /**
     * Creates the specified {@code directory}, if it does not already exist. If the path does exist, it is validated
     * that it is a directory and not a file.
     *
     * @param directory the directory to create
     * @return the created directory
     * @throws IllegalStateException if the {@code directory} path already exists and is not a directory, or if the
     *                               directory cannot be created
     * @throws NullPointerException if the provided {@code directory} is {@code null}
     */
    @Nonnull
    public static Path mkdir(@Nonnull Path directory) {
        requireNonNull(directory, "directory");

        try {
            return Files.createDirectories(directory);
        } catch (IOException e) {
            if (e instanceof FileAlreadyExistsException && Files.isDirectory(directory)) { //Follow links
                //In general File.createDirectories won't throw this, but it will if the target exists and isn't a
                //directory. Unfortunately, Files.createAndCheckIsDirectory doesn't allow symbolic links, which we
                //use, and customers use, to help divide the home directory up
                return directory;
            }

            throw new IllegalStateException("Could not create " + directory.toAbsolutePath(), e);
        }
    }

    /**
     * Creates the specified {@code child} directory beneath the {@code parent}, if it does not already exist. If the
     * path does exist, it is validated that it is a directory and not a file.
     *
     * @param parent the base path for creating the new directory
     * @param child  the path beneath the parent for the new directory
     * @return the created directory
     * @throws IllegalArgumentException if the {@code child} path is blank or empty
     * @throws IllegalStateException if the {@code child} path already exists and is not a directory, or if the
     *                               directory cannot be created
     * @throws NullPointerException if the provided {@code parent} or {@code child} is {@code null}
     */
    @Nonnull
    public static Path mkdir(@Nonnull Path parent, @Nonnull String child) {
        requireNonNull(parent, "parent");
        if (requireNonNull(child, "child").trim().isEmpty()) {
            throw new IllegalArgumentException("A path for the created directory is required");
        }

        return mkdir(parent.resolve(child));
    }

    /**
     * Validates that the specified path is contained within the {@code expectedParent}. This should be used to
     * ensure paths created with user-entered data don't "escape" the specified parent, to prevent path traversal
     * attacks.
     *
     * @param path           the path to validate
     * @param expectedParent the required parent directory
     * @throws IllegalArgumentException if {@code expectedParent} does not exist or is not a directory, or if
     *                                  {@code path} is not contained within it
     * @throws IOException if the real path for either of the provided paths cannot be resolved
     */
    public static void requireWithin(@Nonnull Path path, @Nonnull Path expectedParent) throws IOException {
        if (!isWithin(path, expectedParent)) {
            throw new IllegalArgumentException(path + " is not contained within " + expectedParent);
        }
    }

    /**
     * Simplifies {@link Path#resolve(String) resolving} several subpaths in a single call.
     *
     * @param path  the base path, from which subpaths should be resolved
     * @param first the first subpath, used to enforce that at least a single subpath is provided
     * @param more  zero or more additional subpaths
     * @return the resolved path
     */
    @Nonnull
    public static Path resolve(@Nonnull Path path, @Nonnull String first, @Nonnull String... more) {
        return requireNonNull(path, "path").resolve(Paths.get(first, more));
    }

    /**
     * Sets the file permissions according to the specifications provided in the {@link SetFilePermissionRequest request}.
     * For Unix systems this will map the provided permissions to the corresponding {@link PosixFilePermission}.
     * For Windows systems {@link AclEntry ACL entries} will be put together as follows:
     * <ul>
     *     <li>{@link SetFilePermissionRequest#getOwnerPermissions() Owner permissions}: These will be applied to the
     *     {@link Files#getOwner(Path, java.nio.file.LinkOption...) owner principal}. Also, for convenience,
     *     they will be applied to the {@code Administrator} group (if it exists) since its members can override
     *     ACLs anyway</li>
     *     <li>{@link SetFilePermissionRequest#getWorldPermissions() World permissions}: These will be applied to the
     *     {@code Everyone} group.</li>
     *     <li>{@link SetFilePermissionRequest#getGroupPermissions() Group permissions}: These will be ignored.
     *     In POSIX-compliant OSes files and directories are associated with both a user and a group and
     *     the permission model explicitly provides for setting a group permission <i>agnostic of the group's name</i>.
     *     The ACL model used by Windows requires knowing the name of the group you want to provision.</li>
     * </ul>
     *
     * @param request describes the file permissions to set
     * @throws IllegalArgumentException if the provided {@link SetFilePermissionRequest#getPath path} does not exist,
     *                                  or exists and is not a regular file
     */
    public static void setPermissions(@Nonnull SetFilePermissionRequest request) throws IOException {
        Path path = requireNonNull(request, "request").getPath();
        if (!Files.isRegularFile(path)) {
            throw new IllegalArgumentException("The provided file, " + path + ", does not exist or is not a file");
        }

        if (SystemUtils.IS_OS_WINDOWS) {
            AclFileAttributeView view = Files.getFileAttributeView(path, AclFileAttributeView.class);

            UserPrincipalLookupService principalService = path.getFileSystem().getUserPrincipalLookupService();

            ImmutableList.Builder<AclEntry> entries = ImmutableList.builder();
            Set<FilePermission> ownerPermissions = request.getOwnerPermissions();
            if (!ownerPermissions.isEmpty()) {
                entries.add(createAclEntry(Files.getOwner(path), ownerPermissions));

                //Administrators can override the ACLs so, for convenience, we might as well give them owner permissions
                createAclEntry(principalService, "administrators", ownerPermissions).ifPresent(entries::add);
            }

            Set<FilePermission> worldPermissions = request.getWorldPermissions();
            if (!worldPermissions.isEmpty()) {
                createAclEntry(principalService, "everyone", request.getWorldPermissions()).ifPresent(entries::add);
            }

            view.setAcl(entries.build());
        } else {
            try {
                Files.setPosixFilePermissions(path, toPosixFilePermissions(request));
            }  catch (UnsupportedOperationException e) {
                // If you're not running Windows and you're not POSIX compliant then just throw an IOException warning
                // clients that file permissions could not be set.
                throw new IOException("Failed to update file permissions for " + path.toAbsolutePath(), e);
            }
        }
    }

    /**
     * Gets the {@link Files#size size} of the specified {@link Path}.
     * <p>
     * Like {@link File#length}, if the path does not exist or its attributes cannot be read, {@code 0L} is
     * returned rather than throwing an exception. Use {@link Files#size} directly if an exception is desired.
     *
     * @param path the path to retrieve the size of
     * @return the file's size, or {@code 0L} if the size could not be determined
     */
    public static long size(@Nonnull Path path) {
        requireNonNull(path, "path");

        try {
            return Files.size(path);
        } catch (IOException e) {
            return 0L;
        }
    }

    /**
     * Calculates the <i>total size</i> of the specified path.
     * <p>
     * If the specified {@link Path} denotes a directory, its contents are <i>recursively traversed</i> to
     * compute its overall size. For directories containing a large number of files this can be quite slow,
     * so it should be used with caution.
     * <p>
     * If the specified {@link Path} denotes a symbolic link, its size (generally the length of its target
     * path) is reported. Symbolic links are <i>not</i> followed.
     * <p>
     * If the specified {@link Path} denotes a file, its size is reported.
     *
     * @param path the path to calculate a size for
     * @return the calculated size, or {@code 0L} if the size could not be calculated
     */
    public static long totalSize(@Nonnull Path path) {
        requireNonNull(path, "path");

        try {
            MutableLong size = new MutableLong();

            Files.walkFileTree(path, new SimpleFileVisitor<Path>() {

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    size.add(attrs.size());

                    return FileVisitResult.CONTINUE;
                }
            });

            return size.longValue();
        } catch (IOException ignored) {
            return 0L;
        }
    }

    /**
     * {@link Files#createFile Creates} the specified path, if it doesn't exist, or
     * {@link Files#setLastModifiedTime sets its last modified time} if it does.
     *
     * @param path the path to create or set a last modification time for
     * @throws IOException if a nonexistent file cannot be created, or if the last modification time cannot be
     *                     set for an existing file
     */
    public static void touch(@Nonnull Path path) throws IOException {
        try {
            Files.setLastModifiedTime(path, FileTime.from(Instant.now()));
        } catch (FileNotFoundException | NoSuchFileException e) {
            //If the file doesn't exist, create it. The newly-created file should have the
            //current time for its last modified time, so there's no need to set it again
            Files.createFile(path);
        }
    }

    /**
     * Reads the entire contents of the specified path using the UTF-8 character set.
     * <p>
     * Callers should exercise caution when using this method, as it may result in reading a significant amount
     * of data into memory. Where possible, callers are encouraged to stream file contents instead.
     *
     * @param path the path to read
     * @return the file's contents
     * @throws IOException if the file cannot be read
     */
    @Nonnull
    public static String toString(@Nonnull Path path) throws IOException {
        return toString(path, StandardCharsets.UTF_8);
    }

    /**
     * Reads the entire contents of the specified path using the provided character set.
     * <p>
     * Callers should exercise caution when using this method, as it may result in reading a significant amount
     * of data into memory. Where possible, callers are encouraged to stream file contents instead.
     *
     * @param path    the path to read
     * @param charset the character set to use to decode the file's contents
     * @return the file's contents
     * @throws IOException if the file cannot be read
     */
    @Nonnull
    public static String toString(@Nonnull Path path, @Nonnull Charset charset) throws IOException {
        requireNonNull(path, "path");
        requireNonNull(charset, "charset");

        return new String(Files.readAllBytes(path), charset);
    }

    /**
     * Writes the provided value to the specified path using the UTF-8 character set.
     *
     * @param path    the path to write
     * @param value   the value to write
     * @param options {@link OpenOption}s to control how the path is accessed
     * @return the provided path
     * @throws IOException if the file cannot be written
     */
    @Nonnull
    public static Path write(@Nonnull Path path, @Nonnull String value, @Nonnull OpenOption... options)
            throws IOException {
        return write(path, value, StandardCharsets.UTF_8, options);
    }

    /**
     * Writes the provided value to the specified path using the provided character set.
     *
     * @param path    the path to write
     * @param value   the value to write
     * @param charset the character set to use to encode the value to bytes
     * @param options {@link OpenOption}s to control how the path is accessed
     * @return the provided path
     * @throws IOException if the file cannot be written
     */
    @Nonnull
    public static Path write(@Nonnull Path path, @Nonnull String value, @Nonnull Charset charset,
                             @Nonnull OpenOption... options) throws IOException {
        requireNonNull(path, "path");
        requireNonNull(value, "value");
        requireNonNull(charset, "charset");

        return Files.write(path, value.getBytes(charset), options);
    }

    private static AclEntry createAclEntry(UserPrincipal principal, Set<FilePermission> permissions) {
        return AclEntry.newBuilder()
                .setPermissions(permissions.stream().flatMap(permission -> {
                    switch (permission) {
                        case EXECUTE:
                            return EXECUTE_ACL_PERMISSIONS.stream();
                        case READ:
                            return READ_ACL_PERMISSIONS.stream();
                        default:
                            return WRITE_ACL_PERMISSIONS.stream();
                    }
                }).collect(toImmutableSet()))
                .setPrincipal(principal)
                .setType(AclEntryType.ALLOW)
                .build();
    }

    private static Optional<AclEntry> createAclEntry(UserPrincipalLookupService principalService, String principalName,
                                                     Set<FilePermission> permissions) throws IOException {
        try {
            return of(createAclEntry(principalService.lookupPrincipalByGroupName(principalName), permissions));
        } catch (UserPrincipalNotFoundException e) {
            //Nothing we can do here 'Everyone' and 'Administrators' are default groups in Windows so the customer
            //has a non standard installation
            return empty();
        }
    }

    private static Set<PosixFilePermission> toPosixFilePermissions(SetFilePermissionRequest request) {
        ImmutableSet.Builder<PosixFilePermission> posixPermissions = ImmutableSet.builder();

        request.getOwnerPermissions().stream().map(permission -> {
            switch (permission) {
                case EXECUTE:
                    return PosixFilePermission.OWNER_EXECUTE;
                case READ:
                    return PosixFilePermission.OWNER_READ;
                default:
                    return PosixFilePermission.OWNER_WRITE;
            }
        }).forEach(posixPermissions::add);

        request.getGroupPermissions().stream().map(permission -> {
            switch (permission) {
                case EXECUTE:
                    return PosixFilePermission.GROUP_EXECUTE;
                case READ:
                    return PosixFilePermission.GROUP_READ;
                default:
                    return PosixFilePermission.GROUP_WRITE;
            }
        }).forEach(posixPermissions::add);

        request.getWorldPermissions().stream().map(permission -> {
            switch (permission) {
                case EXECUTE:
                    return PosixFilePermission.OTHERS_EXECUTE;
                case READ:
                    return PosixFilePermission.OTHERS_READ;
                default:
                    return PosixFilePermission.OTHERS_WRITE;
            }
        }).forEach(posixPermissions::add);

        return posixPermissions.build();
    }
}
