package com.atlassian.bitbucket.util;

import com.atlassian.bitbucket.validation.annotation.OptionalString;
import com.atlassian.bitbucket.validation.annotation.RequiredString;
import org.apache.commons.lang3.StringUtils;

/**
 * More precondition checks following the same pattern as Guava's {@code Preconditions}.
 */
public class CustomPreconditions {

    /**
     * Checks whether {@code value} validates as a required string. This method duplicates the logic of
     * the {@link RequiredString RequiredString} annotation and exists only to
     * support cases where annotations cannot be used.
     * 
     * @param value the value to check
     * @throws IllegalArgumentException if {@code value} is blank
     */
    public static void checkRequiredString(String value) {
        if (StringUtils.isBlank(value)) {
            throw new IllegalArgumentException("cannot be blank");
        }
    }

    /**
     * Checks whether {@code value} validates as a required string. This method duplicates the logic of
     * the {@link RequiredString RequiredString} annotation and exists only to
     * support cases where annotations cannot be used.
     *
     * @param value the value to check
     * @param maxLength the maximum allowed string length
     * @throws IllegalArgumentException if {@code value} is blank or longer than {@code maxLength} characters
     */
    public static void checkRequiredString(String value, int maxLength) {
        if (StringUtils.isBlank(value)) {
            throw new IllegalArgumentException("cannot be blank");
        } else if (value.length() > maxLength) {
            throw new IllegalArgumentException("must be between 1 and " + maxLength + " characters");
        }
    }

    /**
     * Checks whether {@code value} validates as an optional string. This method duplicates the logic of
     * the {@link OptionalString OptionalString} annotation and exists only to
     * support cases where annotations cannot be used.
     *
     * @param value the value to check
     * @param maxLength the maximum allowed string length
     * @throws IllegalArgumentException if {@code value} trims to the empty string or longer than {@code maxLength}
     *                                  characters. {@code null} is allowed.
     */
    public static void checkOptionalString(String value, int maxLength) {
        if (value != null) {
            if (value.trim().length() == 0) {
                throw new IllegalArgumentException("cannot be blank");
            } else if (value.length() > maxLength) {
                throw new IllegalArgumentException("must be between 1 and " + maxLength + " characters");
            }
        }
    }
}
