package com.atlassian.bitbucket.topic;

import com.atlassian.bitbucket.event.cluster.ClusterNodeEvent;

import javax.annotation.Nonnull;
import java.io.Serializable;

/**
 * Provides a mechanism for publishing messages to multiple subscribers, both local and cluster-wide. When a listener
 * subscribes to a topic, it receives all messages published to that topic from any node until the listener
 * unsubscribes.
 * <p>
 * Messages are delivered in the order they were published on the node they were published on. If node N publishes
 * messages m1, m2..mn, all listeners receive messages m1, m2..mn in order.
 * <p>
 * Messages are <em>not</em> durable and message delivery is guaranteed only to the listeners that were reachable at
 * the time a message is published. In the case of network partitions, listeners may miss messages. Plugins that rely
 * on {@code TopicService} to handle cluster coherency are encouraged to also listen for the various
 * {@link ClusterNodeEvent cluster node events} to handle nodes joining, rejoining and leaving the cluster.
 */
public interface TopicService {

    /**
     * Returns the existing topic, or creates new topic with the provided settings if it does not yet exist.
     *
     * @param topic the topic name
     * @param settings the topic configuration
     * @param <T> the message type of the topic
     * @return the topic
     *
     * @since 5.3
     */
    @Nonnull
    <T extends Serializable> Topic<T> getTopic(@Nonnull String topic, @Nonnull TopicSettings<T> settings);

    /**
     * Publishes a message to all (both local and remote) listeners of the topic
     *
     * @param topic the topic to publish to
     * @param message the message
     *
     * @deprecated in 5.3 for removal in 6.0. Use {@link #getTopic} to obtain the topic and call {@link Topic#publish}
     *             instead.
     */
    @Deprecated
    <T extends Serializable> void publish(@Nonnull String topic, @Nonnull T message);

    /**
     * Subscribes a provided listener to a topic. The listener will receive callbacks for every message published to
     * the topic after the listener was subscribed.
     * <p>
     * If the topic did not exist, prior to subscribing, it will be created with default
     * {@link TopicSettings settings}.
     *
     * @param topic the topic to subscribe to
     * @param listener the listener that should receive callbacks when a message is published to the topic
     * @param <T> the message type of the topic
     * @return the subscription ID, used to {@link #unsubscribe(String) unsubscribe}.
     *
     * @deprecated in 5.3 for removal in 6.0. Use {@link #getTopic} to obtain the topic and call {@link Topic#subscribe}
     *             instead.
     */
    @Deprecated
    @Nonnull
    <T extends Serializable> String subscribe(@Nonnull String topic, @Nonnull TopicListener<T> listener);

    /**
     * Unsubscribes a listener. The listener will not receive any more messages the topic.
     *
     * @param subscriptionId the subscription ID that was returned by {@link #subscribe(String, TopicListener)}
     * @return {@code true} if the listener was successfully unsubscribed, otherwise {@code false}
     *
     * @deprecated in 5.3 for removal in 6.0. Use {@link #getTopic} to obtain the topic and call
     *             {@link Topic#unsubscribe} instead.
     */
    @Deprecated
    boolean unsubscribe(@Nonnull String subscriptionId);
}
