package com.atlassian.bitbucket.server;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;

/**
 * Core {@link Feature features} provided by the application
 *
 * @since 4.2
 */
public enum StandardFeature implements Feature {

    /**
     * Controls whether users can upload attachments in pull request comments. Disabling this feature will
     * <i>not</i> delete any attachments that have already been uploaded, but will prevent uploading more.
     */
    ATTACHMENTS("attachments"),
    /**
     * Controls whether users will be required to solve a CAPTCHA after a configured number of failed attempts
     * to login. Disabling this feature can reduce the overall security of an instance and is not recommended,
     * but may be necessary in some environments. If this feature is disabled, it is strongly advised that some
     * other aspect of the authentication mechanism (such as a remote LDAP directory) impose a limit on failed
     * attempts in order to prevent attackers from using brute force attacks to compromise accounts.
     */
    AUTH_CAPTCHA("auth.captcha"),
    /**
     * Controls whether diagnostics is enabled
     */
    DIAGNOSTICS("diagnostics"),
    /**
     * Controls whether users can edit repository files in the browser and via REST. Disabling this feature
     * <i>does not</i> prevent plugins from using the Java API to edit files; it only disables editing via
     * the UI and REST.
     *
     * @see com.atlassian.bitbucket.content.ContentService#editFile
     * @since 4.13
     */
    FILE_EDITOR("file.editor"),
    /**
     * Controls whether repositories are allowed to be forked. When this feature is disabled, it overrides any
     * configuration applied to individual repositories.
     *
     * @see com.atlassian.bitbucket.repository.ForkingDisabledException
     * @see com.atlassian.bitbucket.repository.RepositoryService#fork
     */
    FORKS("forks"),
    /**
     * Controls whether new users will be presented with a "Getting Started" page after their first login.
     */
    GETTING_STARTED("getting.started.page"),
    /**
     * Controls whether Data Center Migration archives can be generated on this instance.
     * <p>
     * This is <i>not</i> a {@link #isDataCenter() data center-only} feature, data can be exported from Server
     * installations as well.
     *
     * @since 5.13
     */
    DATA_CENTER_MIGRATION_EXPORT("data.center.migration.export"),
    /**
     * Controls whether Data Center Migration archives can be imported into the instance.
     * <p>
     * This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 5.13
     */
    DATA_CENTER_MIGRATION_IMPORT("data.center.migration.import", true),
    /**
     * Controls whether users can create repositories in their personal projects. Disabling this feature will
     * prevent creating new repositories in, or forking existing repositories to, users' personal projects.
     * To completely disable forking, regardless of the target project's type, the {@link #FORKS} feature should
     * be disabled instead of, or in addition to, this one.
     *
     * @see com.atlassian.bitbucket.repository.PersonalRepositoryDisabledException
     */
    PERSONAL_REPOS("personal.repos"),
    /**
     * Controls whether {@link Project#isPublic() projects} and {@link Repository#isPublic() repositories} can be
     * configured to allow public access. When this feature is disabled, it overrides any configuration applied to
     * individual projects or repositories.
     */
    PUBLIC_ACCESS("public.access"),
    /**
     * Controls whether pull requests can be deleted. When this feature is disabled, pull requests cannot be
     * disabled via plugins using the Java API or using the UI or REST.
     *
     * @see com.atlassian.bitbucket.pull.PullRequestDeletionDisabledException
     * @see com.atlassian.bitbucket.pull.PullRequestService#delete
     * @since 5.1
     */
    PULL_REQUEST_DELETION("pull.request.deletion"),
    /**
     * Controls whether smart mirrors can be configured. This is a {@link #isDataCenter() data center-only} feature.
     * Enabling or disabling this feature has no effect if a data center license is not installed. Disabling this
     * feature prevents the use of smart mirrors even if a data center license <i>is</i> installed.
     */
    SMART_MIRRORS("smart.mirrors", true),
    /**
     * Controls whether users will be prompted to update their timezone after their first login.
     */
    TIME_ZONE_ONBOARDING("user.time.zone.onboarding");

    private final boolean dataCenter;
    private final String key;

    StandardFeature(String key) {
        this(key, false);
    }

    StandardFeature(String key, boolean dataCenter) {
        this.dataCenter = dataCenter;
        this.key = key;
    }

    @Nonnull
    public static StandardFeature fromKey(String value) {
        for (StandardFeature feature : values()) {
            if (feature.getKey().equals(value)) {
                return feature;
            }
        }
        throw new IllegalArgumentException("No Feature is associated with key [" + value + "]");
    }

    @Nonnull
    @Override
    public String getKey() {
        return key;
    }

    @Override
    public boolean isDataCenter() {
        return dataCenter;
    }
}
