package com.atlassian.bitbucket.scope;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.util.Objects;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Describes a repository level scope
 *
 * @since 5.0
 */
public class RepositoryScope implements Scope {

    private final Repository repository;

    public RepositoryScope(@Nonnull Repository repository) {
        this.repository = requireNonNull(repository, "repository");
    }

    public <T> T accept(@Nonnull ScopeVisitor<T> visitor) {
        return visitor.visit(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof RepositoryScope) {
            RepositoryScope that = (RepositoryScope) o;

            return Objects.equals(getRepository(), that.getRepository());
        }

        return false;
    }

    /**
     * @return the project containing the {@link #getRepository repository}
     */
    @Nonnull
    public Project getProject() {
        return repository.getProject();
    }

    /**
     * @return the repository
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    @Nonnull
    @Override
    public Optional<Integer> getResourceId() {
        return Optional.of(repository.getId());
    }

    @Nonnull
    public ScopeType getType() {
        return ScopeType.REPOSITORY;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getRepository());
    }

    @Override
    public String toString() {
        return "RepositoryScope{repository=" + repository.toString() + "}";
    }
}
