package com.atlassian.bitbucket.scm.signed;

import com.atlassian.bitbucket.scm.ScmExtendedCommandFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;

/**
 * Describes a callback for receiving information about signed objects.
 *
 * @see ScmExtendedCommandFactory#signedObjects(SignedObjectsParameters, SignedObjectCallback)
 * @since 5.1
 */
public interface SignedObjectCallback {

    /**
     * Called after the signature information for the final requested object has been provided.
     * <p>
     * Note: If there no objects were requested, this method may be called immediately after
     * {@link #onStart(SignedObjectsContext)} without any calls to any of the other callback methods.
     *
     * @param summary summarizes the request
     * @throws IOException for implementations which perform I/O
     */
    default void onEnd(@Nonnull SignedObjectsSummary summary) throws IOException {
    }

    /**
     * Called when no object was found in the repository for the requested {@code objectId}.
     *
     * @param objectId the requested object ID
     * @throws IOException for implementations which perform I/O
     */
    void onMissing(@Nonnull String objectId) throws IOException;

    /**
     * Called when the requested object was found and contained a signature.
     *
     * @param type the type of object for which the signature and signed content is returned
     * @param objectId the requested object ID
     * @param signature the signature
     * @param signedContent the object content that was signed
     * @throws IOException for implementations which perform I/O
     */
    void onSigned(@Nonnull SignableObjectType type, @Nonnull String objectId, @Nonnull String signature,
                  @Nonnull String signedContent) throws IOException;

    /**
     * Called before the signature information for the first requested object is provided.
     *
     * @param context provides details about the request for signature information is being streamed
     * @throws IOException for implementations which perform I/O
     */
    default void onStart(@Nonnull SignedObjectsContext context) throws IOException {
    }

    /**
     * Called when the requested object was found, but was not signed.
     *
     * @param type the type of object that was found if it's signable, otherwise null
     * @param objectId the requested object ID
     * @throws IOException for implementations which perform I/O
     */
    void onUnsigned(@Nullable SignableObjectType type, @Nonnull String objectId) throws IOException;
}
