package com.atlassian.bitbucket.scm.signed;

import javax.annotation.Nonnull;
import java.util.NoSuchElementException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;

import static java.util.Objects.requireNonNull;

/**
 * {@link SignedObjectIdSource} that you can add items to that blocks until a next item becomes available or
 * {@link #complete()} is called.
 *
 * @since 5.1
 */
public class InteractiveSignedObjectIdSource implements SignedObjectIdSource {

    // Marker that is used to signal that the end of the input has been reached
    private static final String COMPLETE = "\0\0____________COMPLETE____________\0\0";

    private final BlockingQueue<String> items;

    private volatile boolean complete;
    private volatile String next;

    public InteractiveSignedObjectIdSource() {
        items = new LinkedBlockingDeque<>();
    }

    /**
     * Adds an object ID to be {@link #next() returned} by this source.
     *
     * @param objectId the next object ID
     */
    public void add(@Nonnull String objectId) {
        if (complete) {
            throw new IllegalStateException("Already completed");
        }
        items.offer(requireNonNull(objectId, "objectId"));
    }

    /**
     * Marks the source as complete. After this method is called, {@link #hasNext()} will return {@code false} after all
     * previously {@link #add(String) added} object IDs have been returned.
     */
    public void complete() {
        complete = true;
        items.offer(COMPLETE);
    }

    @Override
    public boolean hasNext() {
        while (next == null && !(complete && items.isEmpty())) {
            try {
                next = items.poll(500, TimeUnit.MILLISECONDS);
                //noinspection StringEquality
                if (next == COMPLETE) {
                    next = null;
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        return next != null;
    }

    @Nonnull
    @Override
    public String next() {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }

        String objectId = next;
        next = null;

        return objectId;
    }
}
