package com.atlassian.bitbucket.scm.bulk;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.AbstractCommandParameters;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

/**
 * Parameters for traversing commits in one or more repositories.
 *
 * @see ScmBulkContentCommandFactory#traverseCommits(BulkTraverseCommitsCommandParameters, BulkTraversalCallback)
 * @since 5.11
 */
public class BulkTraverseCommitsCommandParameters extends AbstractCommandParameters {

    private final boolean ignoreMissing;
    private final Set<String> includes;
    private final Set<Repository> alternates;

    private BulkTraverseCommitsCommandParameters(Builder builder) {
        ignoreMissing = builder.ignoreMissing;
        includes = builder.includes.build();
        alternates = builder.alternates.build();

        if (includes.isEmpty()) {
            throw new IllegalStateException("The parameters must include commits to traverse.");
        }
    }

    /**
     * @return alternate repositories to look in for included commits
     */
    @Nonnull
    public Set<Repository> getAlternates() {
        return alternates;
    }

    /**
     * @return included commits to traverse
     */
    @Nonnull
    public Set<String> getIncludes() {
        return includes;
    }

    /**
     * @return whether or not to ignore missing commits.
     */
    public boolean isIgnoringMissing() {
        return ignoreMissing;
    }

    public static class Builder {

        private final ImmutableSet.Builder<Repository> alternates;
        private final ImmutableSet.Builder<String> includes;

        private boolean ignoreMissing;

        public Builder() {
            alternates = ImmutableSet.builder();
            includes = ImmutableSet.builder();
        }

        @Nonnull
        public BulkTraverseCommitsCommandParameters build() {
            return new BulkTraverseCommitsCommandParameters(this);
        }

        /**
         * @param value an alternate repository to look in for included commits
         * @return the builder
         */
        @Nonnull
        public Builder alternate(@Nullable Repository value) {
            if (value != null) {
                alternates.add(value);
            }

            return this;
        }


        /**
         * @param values alternate repositories to look in for included commits
         * @return the builder
         */
        @Nonnull
        public Builder alternates(@Nullable Iterable<Repository> values) {
            addIf(Objects::nonNull, alternates, values);

            return this;
        }

        /**
         * @param value  an alternate repository to look in for included commits
         * @param values zero or more alternate repositories to look in for included commits
         * @return the builder
         */
        @Nonnull
        public Builder alternates(@Nullable Repository value, @Nullable Repository... values) {
            addIf(Objects::nonNull, alternates, value, values);

            return this;
        }

        /**
         * @param value boolean value indicating whether or not to ignore missing commits
         * @return the builder
         */
        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoreMissing = value;

            return this;
        }

        /**
         * Specify commits to include in the traversal. These should be commit hashes rather than ref names - if a ref
         * name is provided it will only be resolved for the primary repository and not any
         * {@link #getAlternates() alternates}  specified.
         *
         * @param values commits to include in the traversal
         * @return the builder
         */
        @Nonnull
        public Builder include(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, includes, values);

            return this;
        }

        /**
         * Specify commits to include in the traversal. These should be commit hashes rather than ref names - if a ref
         * name is provided it will only be resolved for the primary repository and not any
         * {@link #getAlternates() alternates}  specified.
         *
         * @param value  a commit to include in the traversal
         * @param values zero or more commits to include in the traversal
         * @return the builder
         */
        @Nonnull
        public Builder include(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, includes, value, values);

            return this;
        }
    }
}
