package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.ChangeContext;
import com.atlassian.bitbucket.content.ChangesRequest;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

public class ChangesCommandParameters extends AbstractChangesCommandParameters {

    private final String sinceId;
    private final String untilId;

    private ChangesCommandParameters(Builder builder) {
        super(builder);

        untilId = builder.untilId;
        sinceId = builder.sinceId;
    }

    @Nullable
    public String getSinceId() {
        return sinceId;
    }

    @Nonnull
    public String getUntilId() {
        return untilId;
    }

    public boolean hasSinceId() {
        return StringUtils.isNotBlank(sinceId);
    }

    /**
     * Assembles a {@link ChangeContext} from these parameters.
     *
     * @return a {@link ChangeContext} describing these parameters
     */
    @Nonnull
    public ChangeContext toContext() {
        return toContext(new ChangeContext.Builder(getSinceId(), getUntilId()));
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private String sinceId;
        private String untilId;

        public Builder() {
        }

        public Builder(@Nonnull ChangesRequest request) {
            sinceId = requireNonNull(request, "request").getSinceId();
            untilId = request.getUntilId();
        }

        @Nonnull
        public ChangesCommandParameters build() {
            if (StringUtils.isBlank(untilId)) {
                throw new IllegalStateException("An \"until\" ID is required to retrieve changes");
            }

            return new ChangesCommandParameters(this);
        }

        @Nonnull
        public Builder sinceId(@Nullable String value) {
            sinceId = value;

            return this;
        }

        @Nonnull
        public Builder untilId(@Nonnull String value) {
            //Nullability is enforced in build(); "value" is annotated @Nonnull to help callers avoid errors
            untilId = value;

            return this;
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
