package com.atlassian.bitbucket.repository;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * A callback for streaming {@link Tag tags}.
 */
public interface TagCallback {

    /**
     * Called after the final {@link #onTag tag} has been streamed.
     * <p>
     * Note: If there were no tags, this method may be called immediately after {@link #onStart(TagContext)} without
     * any calls to {@link #onTag(Tag)}.
     *
     * @param summary summarizes the request and the streamed tags
     * @throws IOException May be thrown by implementations which perform I/O
     */
    default void onEnd(@Nonnull TagSummary summary) throws IOException {
    }

    /**
     * Called before the <i>first</i> {@link #onTag tag} is streamed.
     *
     * @param context provides details about the request for which tags are being streamed
     * @throws IOException May be thrown by implementations which perform I/O
     */
    default void onStart(@Nonnull TagContext context) throws IOException {
    }

    /**
     * Called to process a {@link Tag tag}.
     *
     * @param tag the {@link Tag tag} to be handled by the callback
     * @return {@code true} if further tags should be streamed; otherwise, {@code false} to stop streaming
     * @throws IOException May be thrown by implementations which perform I/O
     */
    default boolean onTag(@Nonnull Tag tag) throws IOException {
        return true;
    }
}
