package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A request to generate a repository's clone URLs
 */
public class RepositoryCloneLinksRequest {

    private final Repository repository;
    private final String protocolName;
    private final ApplicationUser user;
    private final boolean useCurrentUser;

    private RepositoryCloneLinksRequest(@Nonnull Builder builder) {
        checkNotNull(builder, "builder");
        //This should not be possible from the builder
        checkArgument(!builder.useCurrentUser || builder.user == null,
                "A user cannot be supplied if the current authenticated user is to be used for clone link generation");

        this.repository = checkNotNull(builder.repository, "repository");
        this.protocolName = builder.protocolName;
        this.user = builder.user;
        this.useCurrentUser = builder.useCurrentUser;
    }

    /**
     * @return the repository that the resulting links should be for
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * @return the optional protocol that the resulting links should match
     */
    @Nullable
    public String getProtocolName() {
        return protocolName;
    }

    /**
     * @return the user to be used when generating the clone link or null if no user is to be used or the
     * {@link #isUseCurrentUser() user for the current request} is to be used
     */
    @Nullable
    public ApplicationUser getUser() {
        return user;
    }

    /**
     * @return true if the currently authenticated user is to be used, false otherwise. When building this object if
     * {@link Builder#user(ApplicationUser)} was called then the currently authenticated user will not be used
     */
    public boolean isUseCurrentUser() {
        return useCurrentUser;
    }

    /**
     * Builds {@link RepositoryCloneLinksRequest}
     */
    public static class Builder {

        private Repository repository;
        private String protocolName;
        private ApplicationUser user;
        private boolean useCurrentUser = true;

        public Builder protocol(@Nullable String value) {
            protocolName = value;
            return this;
        }

        public Builder repository(@Nonnull Repository value) {
            repository = value;
            return this;
        }

        public Builder user(@Nullable ApplicationUser value) {
            useCurrentUser = false;
            user = value;

            return this;
        }

        @Nonnull
        public RepositoryCloneLinksRequest build() {
            return new RepositoryCloneLinksRequest(this);
        }
    }
}
