package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A request to create a {@link com.atlassian.bitbucket.repository.Ref ref}.
 */
public abstract class AbstractCreateRefRequest {

    private final String message;
    private final String name;
    private final Repository repository;
    private final String startPoint;

    protected AbstractCreateRefRequest(AbstractBuilder builder) {
        message = builder.message;
        name = builder.name;
        repository = builder.repository;
        startPoint = builder.startPoint;
    }

    /**
     * @return the message to add to the tag
     */
    @Nullable
    public String getMessage() {
        return message;
    }

    /**
     * @return the name of the tag to create
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * @return the repository in which the tag must be created
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * @return the commit id to tag
     */
    @Nonnull
    public String getStartPoint() {
        return startPoint;
    }

    public abstract static class AbstractBuilder<B extends AbstractBuilder, R extends AbstractCreateRefRequest> extends BuilderSupport {

        private final String name;
        private final Repository repository;
        private final String startPoint;

        private String message;

        protected AbstractBuilder(@Nonnull Repository repository, @Nonnull String name, @Nonnull String startPoint) {
            this.repository = checkNotNull(repository, "repository");
            this.name = checkNotBlank(name, "name");
            this.startPoint = checkNotBlank(startPoint, "startPoint");
        }

        @Nonnull
        public abstract R build();

        @Nonnull
        public B message(@Nullable String value) {
            message = StringUtils.trimToNull(value);

            return self();
        }

        @Nonnull
        public abstract B self();
    }
}
