package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.event.pull.PullRequestMergedEvent;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * Describes a request to {@link PullRequestService#merge(PullRequestMergeRequest) merge} a {@link PullRequest}.
 */
public class PullRequestMergeRequest extends AbstractPullRequestRequest {

    private final boolean autoSubject;
    private final Map<String, Object> context;
    private final String message;
    private final String strategyId;
    private final int version;

    private PullRequestMergeRequest(Builder builder) {
        super(builder);

        autoSubject = builder.autoSubject;
        context = builder.context.build();
        message = builder.message;
        strategyId = builder.strategyId;
        version = builder.version;
    }

    /**
     * Retrieves additional supplementary context for the merge request. This context is passed <i>as-is</i> to
     * all of the {@code MergeRequestCheck}s run before the merge is performed and is also included in the
     * {@link PullRequestMergedEvent PullRequestMergedEvent} raised after the merge
     * is completed.
     *
     * @return a map of additional context, which may be empty but is never {@code null}
     */
    @Nonnull
    public Map<String, Object> getContext() {
        return context;
    }

    /**
     * Retrieves the commit message to use when merging the pull request.
     * <p>
     * If {@link #isAutoSubject()} is {@code true}, the system will prepend an auto-generated subject to the message
     * provided here. Otherwise, if it's {@code false}, the provided message will be used as-is.
     * <p>
     * While this is part of the API for merging pull requests, the underlying SCM performing the merge ultimately
     * decides the exact shape of the final commit message. For example, when performing a <i>fast-forward</i> "merge"
     * in Git, any commit message provided here is <i>ignored</i> because the "merge" does not actually create a new
     * commit. Additionally, the SCM may, at its own discretion, augment the provided commit message with more detail.
     * For example, the Git SCM appends the summary from each merged commit to the message.
     *
     * @return the commit message for the pull request
     */
    @Nullable
    public String getMessage() {
        return message;
    }

    /**
     * Retrieves the {@link PullRequestMergeStrategy#getId ID} of the strategy to use when merging the pull request.
     * If a strategy has not been specified, the {@link PullRequestMergeConfig#getDefaultStrategy default strategy}
     * for the {@link PullRequest#getToRef target repository} is used. Only {@link PullRequestMergeStrategy#isEnabled
     * enabled} strategies may be requested.
     *
     * @return the strategy to use when merging the pull request, or {@code null} to use the default
     * @since 4.9
     */
    @Nullable
    public String getStrategyId() {
        return strategyId;
    }

    /**
     * Retrieves the expected {@link PullRequest#getVersion version} for the pull request. If the pull request's
     * actual version does not match, a {@link PullRequestOutOfDateException} will be thrown to indicate that the
     * merge was requested on stale data.
     *
     * @return the pull request version to merge
     */
    public int getVersion() {
        return version;
    }

    /**
     * Retrieves a flag indicating whether the system should prepend an auto-generated subject to the provided
     * {@link #getMessage message}. If no message is provided, this will always be {@code true}.
     *
     * @return {@code true} if no message was provided, or if an auto-generated subject should be prepended to
     *         it; otherwise, {@code false} to use the provided message as-is
     * @since 5.7
     */
    public boolean isAutoSubject() {
        return autoSubject || isBlank(message);
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private final ImmutableMap.Builder<String, Object> context;
        private final int version;

        private boolean autoSubject;
        private String message;
        private String strategyId;

        public Builder(@Nonnull PullRequest pullRequest) {
            super(pullRequest);

            this.version = pullRequest.getVersion();

            autoSubject = true;
            context = ImmutableMap.builder();
        }

        public Builder(int repositoryId, long pullRequestId, int version) {
            super(repositoryId, pullRequestId);

            this.version = version;

            autoSubject = true;
            context = ImmutableMap.builder();
        }

        /**
         * @param value {@code true} to prepend an auto-generated subject to the {@link #message(String) message};
         *              otherwise, {@code false} to use the message as-is
         * @return {@code this}
         * @since 5.7
         */
        @Nonnull
        public Builder autoSubject(boolean value) {
            autoSubject = value;

            return self();
        }

        @Nonnull
        public PullRequestMergeRequest build() {
            return new PullRequestMergeRequest(this);
        }

        @Nonnull
        public Builder context(@Nonnull String key, @Nonnull Object value) {
            context.put(key, value);

            return self();
        }

        @Nonnull
        public Builder context(@Nonnull Map<String, Object> value) {
            context.putAll(value);

            return self();
        }

        @Nonnull
        public Builder message(@Nullable String value) {
            message = value;

            return self();
        }

        /**
         * @param value the ID of the strategy to use when merging the pull request, or {@code null} to use the
         *              {@link PullRequest#getToRef target repository}'s default strategy
         * @return {@code this}
         * @since 4.9
         */
        @Nonnull
        public Builder strategyId(@Nullable String value) {
            strategyId = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
