package com.atlassian.bitbucket.project;

import com.atlassian.bitbucket.AuthorisationException;
import com.atlassian.bitbucket.IntegrityException;
import com.atlassian.bitbucket.ServiceException;
import com.atlassian.bitbucket.avatar.AvatarSupplier;
import com.atlassian.bitbucket.avatar.CacheableAvatarSupplier;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

/**
 * Creates, updates and deletes projects.
 */
public interface ProjectService extends ProjectSupplier {

    /**
     * Creates a new {@link Project} and sets its avatar, if provided.
     * <p>
     * Note: Both the project key and name must be unique.
     *
     * @param request describes the project to create
     * @return the created project
     * @throws ProjectCreationCanceledException if creation is canceled by an event listener
     * @throws ServiceException if the specified {@link ProjectCreateRequest#getKey()}
     *                                                        or {@link ProjectCreateRequest#getName()} is not unique
     */
    @Nonnull
    Project create(@Nonnull ProjectCreateRequest request);

    /**
     * Deletes the specified {@link Project project}, if it exists.
     * <p>
     * {@link PersonalProject Personal projects} <i>cannot be deleted</i>. Additionally, if the project contains any
     * {@link Repository repositories}, they must all be deleted before the project
     * can be deleted.
     *
     * @param project the project to delete
     * @return {@code true} if the project was deleted; otherwise {@code false} if the project did not exist
     * @throws ProjectDeletionCanceledException if deletion is canceled by an event listener
     * @throws IntegrityException if the project is {@link PersonalProject personal},
     *                                                          or has repositories
     * @throws ServiceException if the project exists and could not be deleted
     */
    boolean delete(Project project);

    /**
     * Retrieves a page of projects which are visible for the current user.
     * <p>
     * <b>Note</b>: {@link PersonalProject Personal projects} are <i>never</i> included in the returned page.
     *
     * @param pageRequest the bounds of the page
     * @return a page containing 0 or more {@link Project projects} to which the current user has access
     */
    @Nonnull
    Page<Project> findAll(@Nonnull PageRequest pageRequest);

    /**
     * Retrieves the {@link Project#getKey() keys} for all projects the current user can see.
     * <p>
     * <b>Note</b>: {@link PersonalProject Personal projects} keys are <i>never</i> included in the returned list.
     * <p>
     * <b>Warning</b>: This mechanism is not paged. For systems with large numbers of projects, the returned list may
     * be very large. For callers wishing to iterate over all existing projects, {@link #findAll(PageRequest)} is the
     * correct mechanism to use.
     *
     * @return an unpaged list of keys for <i>all</i> projects the current user can see
     */
    @Nonnull
    List<String> findAllKeys();

    /**
     * Retrieves the current avatar for the specified project at a requested size. Avatars are square, so the size
     * provided here is used as both height and width for the avatar.
     * <p>
     * The requested size will be normalised to fall within a well-defined set sizes. The supported sizes are:
     * <ul>
     *     <li>256</li>
     *     <li>128</li>
     *     <li>96</li>
     *     <li>64</li>
     *     <li>48</li>
     * </ul>
     * Any size larger than 256 will be normalised down to 256, and any size smaller than 48 will be normalised up to
     * 48. Otherwise, sizes are normalised to the next size up, where they don't match exactly: 56 will be normalised
     * to 64, 100 will be normalised to 128, and so on.
     *
     * @param projectId the {@link Project#getId() ID} of the project to retrieve the avatar for
     * @param size      the desired height and width for the avatar
     * @return a supplier which can be used to access the requested avatar
     */
    @Nonnull
    CacheableAvatarSupplier getAvatar(int projectId, int size);

    /**
     * Retrieve a {@link Project} by its {@link Project#getId() ID}.
     *
     * @param id the {@link Project#getId() ID} of the project to retrieve
     * @return the project instance or {@code null} if no such project exists
     * @throws AuthorisationException if the current user does not have permission to
     *                                                              access the requested project
     */
    @Nullable
    @Override
    Project getById(int id);

    /**
     * Retrieves a {@link Project} by its {@link Project#getKey() key}.
     *
     * @param key the {@link Project#getKey() key} of the project to retrieve
     * @return the keyed project, or {@code null} if the key does not match an existing project
     * @throws AuthorisationException if the current user does not have permission to
     *                                                              access the requested project
     */
    @Nullable
    @Override
    Project getByKey(@Nonnull String key);

    /**
     * Retrieves a {@link Project} by its {@link Project#getName() name}.
     *
     * @param name the {@link Project#getName() name} of the project to retrieve
     * @return the named project, or {@code null} if the name does not match an existing project
     * @throws AuthorisationException if the current user does not have permission to
     *                                                              access the requested project
     */
    @Nullable
    @Override
    Project getByName(@Nonnull String name);

    /**
     * Searches for {@link Project projects} which match the provided {@link ProjectSearchRequest criteria}.
     *
     * @param searchRequest a criteria describing the projects to return
     * @param pageRequest   the bounds of the page
     * @return a page containing 0 or more {@link Project projects} which match the provided
     *         {@link ProjectSearchRequest criteria}
     * @see ProjectSearchRequest
     */
    @Nonnull
    Page<Project> search(@Nonnull ProjectSearchRequest searchRequest, @Nonnull PageRequest pageRequest);

    /**
     * Update a project's data with new values. Project {@link Project#getKey() keys} and {@link Project#getName()
     * names} must be unique, so if the provided values match those of another {@link Project} an exception will be
     * thrown.
     * <p>
     * {@link PersonalProject Personal projects} <i>cannot be updated</i>. Their {@link Project#getKey() keys},
     * {@link Project#getName() names} and {@link Project#getDescription() descriptions} are managed by the system.
     *
     * @param request describes the project to update and the updates to apply
     * @return the updated project instance
     * @throws javax.validation.ConstraintViolationException if the updated key or name collide with another project
     * @throws IntegrityException if the project is {@link PersonalProject personal}
     * @throws ProjectModificationCanceledException if modification is canceled by an event listener
     */
    @Nonnull
    Project update(@Nonnull ProjectUpdateRequest request);

    /**
     * Updates the specified project's avatar, replacing it with the one contained in the provided
     * {@link AvatarSupplier supplier}.
     * <p>
     * Previous avatars <i>are not retained</i>. When a project's avatar is updated, the previous avatar is removed.
     * To reuse a previous avatar, it must be provided again.
     * <p>
     * {@link PersonalProject Personal project} avatars <i>cannot be updated</i>. Their avatars are drawn from their
     * {@link PersonalProject#getOwner() owning users}. If Gravatar support is enabled, personal project avatars may
     * be updated by changing the user's avatar in Gravatar. Otherwise, they may not be updated at all.
     *
     * @param projectId the ID of the project to update the avatar for
     * @param supplier  a supplier providing access to the new avatar to use
     * @throws IntegrityException if the project is {@link PersonalProject personal}
     */
    void updateAvatar(int projectId, @Nonnull AvatarSupplier supplier);

    /**
     * Updates the specified project's avatar, replacing it with the one contained in the provided data URI. The
     * data URI is required to contain Base64-encoded image data, and should be in the format:
     * <code>
     *     data:(content type, e.g. image/png);base64,(data)
     * </code>
     * If the data is not Base64-encoded, or if a character set is defined in the URI, it will be rejected.
     * <p>
     * Previous avatars <i>are not retained</i>. When a project's avatar is updated, the previous avatar is removed.
     * To reuse a previous avatar, it must be provided again.
     * <p>
     * {@link PersonalProject Personal project} avatars <i>cannot be updated</i>. Their avatars are drawn from their
     * {@link PersonalProject#getOwner() owning users}. If Gravatar support is enabled, personal project avatars may
     * be updated by changing the user's avatar in Gravatar. Otherwise, they may not be updated at all.
     *
     * @param projectId the ID of the project to update the avatar for
     * @param uri       a data URI containing a Base64-encoded avatar image
     * @throws IntegrityException if the project is {@link PersonalProject personal}
     */
    void updateAvatar(int projectId, @Nonnull String uri);
}
