package com.atlassian.bitbucket.permission;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.util.function.Predicate;

/**
 * Factory for creation permission-predicates
 */
public interface PermissionPredicateFactory {

    /**
     * @return a {@link Project} predicate for project accessibility
     * @see PermissionService#isProjectAccessible(Project)
     */
    @Nonnull
    Predicate<Project> createProjectAccessiblePredicate();

    /**
     * @param permission the permission in question
     * @return a {@link Project} predicate for given {@link Permission}
     * @see PermissionService#hasProjectPermission(Project, Permission)
     */
    @Nonnull
    Predicate<Project> createProjectPermissionPredicate(@Nonnull Permission permission);

    /**
     * Retrieves a {@code Predicate} which applies a check to ensure the current user can access {@link PullRequest}s.
     * Note: Pull requests are never accessible to anonymous users.
     *
     * @return a {@link PullRequest} predicate for pull request accessibility
     * @since 5.11
     */
    @Nonnull
    Predicate<PullRequest> createPullRequestAccessiblePredicate();

    /**
     * Retrieves a {@code Predicate} which applies a check to ensure the current user has the specified permission with
     * regards to {@link PullRequest}s. This predicate may be used for filtering rows from a DAO, for example.
     *
     * @param permission the permission to check for
     * @return a predicate which will check for the specified permission on pull requests
     */
    @Nonnull
    Predicate<PullRequest> createPullRequestPermissionPredicate(@Nonnull Permission permission);

    /**
     * @return a {@link Repository} predicate for repository accessibility
     * @see PermissionService#isRepositoryAccessible(Repository)
     */
    @Nonnull
    Predicate<Repository> createRepositoryAccessiblePredicate();

    /**
     * @param permission the permission in question
     * @return a {@link Repository} predicate for given {@link Permission}
     * @see PermissionService#hasRepositoryPermission(Repository, Permission)
     */
    @Nonnull
    Predicate<Repository> createRepositoryPermissionPredicate(@Nonnull Permission permission);
}
