package com.atlassian.bitbucket.permission;

import com.atlassian.bitbucket.Product;
import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Map;
import java.util.Set;

/**
 * Enumerates the available permissions and describes what they are used to protect.
 */
public enum Permission {

    /**
     * Allows access to change account configuration for a given user such as SSH keys, GPG keys, personal tokens,
     * and password.
     * <p>
     * This permission can not be granted to any user/group it is implied for any user for their own account. This
     * permission is not implied for personal tokens.
     * <p>
     * Users with SYS_ADMIN gains this permission for all users. Users with ADMIN gains this for all users except
     * SYS_ADMIN users.
     *
     * @since 5.5
     */
    USER_ADMIN(11, 200, ImmutableSet.of(Integer.class, ApplicationUser.class), I18nArgs.EMPTY) {
        @Override
        public boolean isGrantable() {
            return false;
        }
    },
    /**
     * Allow view access to a project.
     * <p>
     * This permission cannot be granted to a user/group. It is an implied permission for any user which has
     * at least read access one or more repositories within the project
     */
    PROJECT_VIEW(10, 500, ImmutableSet.of(Integer.class, Project.class), I18nArgs.EMPTY) {
        @Override
        public boolean isGrantable() {
            return false;
        }
    },
    /**
     * Allows read access to a repository.
     * <p>
     * This allows cloning and pulling changes from a repository, adding comments and declining pull requests
     * that target the repository. It also allows creating pull requests if the user has the permission
     * on <em>both</em> the source and target repository.
     */
    REPO_READ(0, 1000, ImmutableSet.of(Integer.class, Repository.class), I18nArgs.EMPTY, PROJECT_VIEW),
    /**
     * Allows write access to a repository.
     * <p>
     * In addition to the permissions already granted by {@link #REPO_READ}, this allows pushing changes
     * to a repository and merging pull requests targeting the repository.
     */
    REPO_WRITE(1, 3000, ImmutableSet.of(Integer.class, Repository.class), I18nArgs.EMPTY, REPO_READ),
    /**
     * Allows to administer a repository.
     * <p>
     * In addition to the permissions already granted by {@link #REPO_WRITE}, this allows accessing and updating
     * the configuration of the repository, such as adding or revoking branch permissions, adding or revoking other
     * repository permissions, renaming or deleting the repository.
     */
    REPO_ADMIN(8, 5000, ImmutableSet.of(Integer.class, Repository.class), I18nArgs.EMPTY, REPO_WRITE),
    /**
     * Allows read access to a project and any repository it contains.
     * <p>
     * This allows listing and viewing <em>all</em> the repositories in a project. It also grants
     * {@link #REPO_READ} to all the repositories in the project.
     */
    PROJECT_READ(2, 2000, ImmutableSet.of(Integer.class, Project.class), I18nArgs.EMPTY, REPO_READ),
    /**
     * Allows write access to a project and any repository it contains.
     * <p>
     * In addition to the permissions already granted by {@link #PROJECT_READ}, this also grants
     * {@link #REPO_WRITE} on all the repositories in the project.
     */
    PROJECT_WRITE(3, 4000, ImmutableSet.of(Integer.class, Project.class), I18nArgs.EMPTY, PROJECT_READ, REPO_WRITE),
    /**
     * Allows administrative access to a project and any repository it contains.
     * <p>
     * In addition to the permissions already granted by {@link #PROJECT_WRITE}, this allows accessing and updating
     * the configuration of the project itself (such as adding other project permissions, configuring the project's
     * avatar, renaming or deleting the project). This also grants {@link #REPO_ADMIN} on all the repositories in
     * the project.
     */
    PROJECT_ADMIN(4, 6000, ImmutableSet.of(Integer.class, Project.class), I18nArgs.EMPTY, PROJECT_WRITE, REPO_ADMIN),
    /**
     * Allows access to the application.
     * <p>
     * This allows the user to authenticate and counts their account towards the license limit.
     */
    LICENSED_USER(9, 0, I18nArgs.PRODUCT),
    /**
     * Allows project creation.
     * <p>
     * This allows the user to create new projects.
     */
    PROJECT_CREATE(5, 7000, I18nArgs.EMPTY, LICENSED_USER),
    /**
     * Allows access to common administration tasks, such as granting global permissions.
     * <p>
     * This grants overall admin access, implying {@link #PROJECT_ADMIN} to all the projects.
     */
    ADMIN(6, 9000, I18nArgs.PRODUCT, PROJECT_CREATE, PROJECT_ADMIN) {
        @Override
        public boolean isGrantableToAll() {
            return false;
        }
    },
    /**
     * Allows access to advanced administration tasks.
     * <p>
     * In addition to the permissions already granted by {@link #ADMIN}, this grants the user full access to
     * the restricted admin functions, such as enabling SSH access, updating the license or migrating databases.
     */
    SYS_ADMIN(7, 10000, I18nArgs.PRODUCT, ADMIN, USER_ADMIN) {
        @Override
        public boolean isGrantableToAll() {
            return false;
        }
    };
    //NOTE: next permission id is 12.

    /** Maps from {@link #getId() IDs} to their permissions. */
    private static Map<Integer, Permission> idToPermissionMap;
    /** Maps from {@link #getWeight() weights} to their permissions. */
    private static Map<Integer, Permission> weightToPermissionMap;

    private final int id;
    private final PermissionI18n i18n;
    private final Set<Permission> inheritedPermissions;
    private final Set<Class<?>> resourceTypes;
    private final int weight;

    // computed fields
    private Set<Permission> implyingPermissions;
    private Set<Permission> inheritingPermissions;

    /**
     * Constructor for resource permissions.
     *
     * @param id permission ID
     * @param weight permission weight
     * @param resourceTypes resource types this permission applies to
     * @param i18nArguments i18n arguments
     * @param inherited inherited permissions
     */
    Permission(int id, int weight, Set<Class<?>> resourceTypes, Object[] i18nArguments, Permission... inherited) {
        this.id = id;
        this.i18n = new PermissionI18n(this, i18nArguments);
        this.resourceTypes = resourceTypes;
        this.weight = weight;

        ImmutableSet.Builder<Permission> builder = ImmutableSet.builder();
        for (Permission p : inherited) {
            appendInheritedPermissions(builder, p);
        }
        inheritedPermissions = builder.build();
    }

    /**
     * Constructor for global permissions. Global permissions are not associated with any resource types.
     *
     * @param id permission ID
     * @param weight permission weight
     * @param i18nArguments i18n arguments
     * @param inherited inherited permissions
     */
    Permission(int id, int weight, Object[] i18nArguments, Permission... inherited) {
        this(id, weight, Collections.<Class<?>>emptySet(), i18nArguments, inherited);
    }


    /**
     * Gets a permission by its id.
     *
     * @param id id of the permission
     * @return the permission
     * @throws IllegalArgumentException if no Permission is associated with the given {@code id}
     */
    //This method is used by the GenericEnumUserType to map from the ID stored in the database to the enumeration entry
    @Nonnull
    public static Permission fromId(int id) {
        if (idToPermissionMap == null) {
            ImmutableMap.Builder<Integer, Permission> builder = ImmutableMap.builder();
            for (Permission permission : values()) {
                builder.put(permission.getId(), permission);
            }
            idToPermissionMap = builder.build();
        }
        Permission permission = idToPermissionMap.get(id);
        if (permission == null) {
            throw new IllegalArgumentException("No Permission is associated with ID [" + id + "]");
        }
        return permission;
    }

    /**
     * Gets a permission by its weight.
     *
     * @param weight weight of the permission
     * @return the permission, or {@code null} if no permission matches the weight
     */
    @Nullable
    public static Permission fromWeight(int weight) {
        if (weightToPermissionMap == null) {
            ImmutableMap.Builder<Integer, Permission> builder = ImmutableMap.builder();
            for (Permission permission : values()) {
                builder.put(permission.getWeight(), permission);
            }
            weightToPermissionMap = builder.build();
        }
        return weightToPermissionMap.get(weight);
    }

    /**
     * Gets all the global permissions.
     *
     * @see #isGlobal()
     */
    @Nonnull
    public static Set<Permission> getGlobalPermissions() {
        return Sets.filter(EnumSet.allOf(Permission.class), Permission::isGlobal);
    }

    /**
     * Gets all the permissions associated with a resource.
     * <p>
     * For example, {@code getPermissionsOn(Project.class)} returns all the permissions that applies
     * to the resource, <em>excluding {@link #isGlobal() global} permissions</em>.
     *
     * @param resourceClass resource the permission must be applicable to
     * @see #isResource(Class)
     */
    @Nonnull
    public static Set<Permission> getPermissionsOn(final Class<?> resourceClass) {
        return Sets.filter(EnumSet.allOf(Permission.class), permission -> permission.isResource(resourceClass));
    }

    /**
     * Gets the permission with the maximum {@link #getWeight() weigth}.
     *
     * @param p1 permission 1 (can be {@code null})
     * @param p2 permission 2 (can be {@code null})
     * @return the permission with the maximum weight
     */
    @Nullable
    public static Permission max(Permission p1, Permission p2) {
        if (p1 == null) {
            return p2;
        }
        return (p2 == null || p1.getWeight() > p2.getWeight()) ? p1 : p2;
    }

    /**
     * @return i18n-ed description of this permission suitable for user interfaces.
     */
    @Nonnull
    public PermissionI18n getI18n() {
        return i18n;
    }

    /**
     * @return the id of this permission
     */
    public int getId() {
        return id;
    }

    /**
     * Get the set of permissions that inherit this permission (excluding this permission).
     */
    @Nonnull
    public Set<Permission> getImplyingPermissions() {
        if (implyingPermissions == null) {
            Set<Permission> inheriting = EnumSet.noneOf(Permission.class);
            inheriting.addAll(getInheritingPermissions());
            inheriting.remove(this);
            implyingPermissions = Collections.unmodifiableSet(inheriting);
        }
        return implyingPermissions;
    }

    /**
     * Gets all permissions this permission inherits.
     */
    @Nonnull
    public Set<Permission> getInheritedPermissions() {
        return inheritedPermissions;
    }

    /**
     * Gets all permissions that inherit this permission (including this permission).
     */
    @Nonnull
    public Set<Permission> getInheritingPermissions() {
        if (inheritingPermissions == null) {
            Set<Permission> perms = EnumSet.of(this);
            for (Permission p : Permission.values()) {
                if (p.getInheritedPermissions().contains(this)) {
                    perms.add(p);
                }
            }
            inheritingPermissions = Collections.unmodifiableSet(perms);
        }
        return inheritingPermissions;
    }

    /**
     * Return all resource types that this permission applies to. This could be domain objects, as well as numeric
     * values representing resource IDs.
     *
     * @return resources types that this permission applies to, or an empty set for
     * {@link #isGlobal() global permissions}.
     * @see #isGlobal()
     * @see #isResource()
     */
    @Nonnull
    public Set<Class<?>> getResourceTypes() {
        return resourceTypes;
    }

    /**
     * Retrieves the weight of this permission relative to other permissions.
     * <p>
     * Higher weight implies the permission has precedence over its lesser counterpart(s).
     * Weight can be used to perform an in-order traversal of the permission hierarchy.
     *
     * @return the weight of this permission
     */
    public int getWeight() {
        return weight;
    }

    /**
     * Indicates whether this {@code Permission} can be granted globally.
     * <p>
     * Global permissions applies to all resources.
     *
     * @return {@code true} if this permission applies globally to all resources; or
     *         {@code false} if this permission applied to specific resource(s)
     */
    public boolean isGlobal() {
        return resourceTypes.isEmpty();
    }

    /**
     * @return {@code true} if the permission can be granted to a user, group or all users,
     *         {@code false} otherwise
     */
    public boolean isGrantable() {
        return true;
    }

    /**
     * Indicates whether this permission may be granted to all users,
     * or if it must be granted to users or groups individually.
     *
     * @return {@code true} if the permission may be blanket granted to all users; or
     *         {@code false} if it may only be granted individually to users or groups
     */
    public boolean isGrantableToAll() {
        return isGrantable();
    }

    /**
     * Indicates whether this {@code Permission} only applies to specific resource(s),
     * such as projects and repositories.
     * <p>
     * This is the logical negation of {@link #isGlobal()}; a permission cannot be both global and resource.
     *
     * @return {@code true} if this permission can be applied to a specific resource; or
     *         {@code false} if this is a {@link #isGlobal() global} permission
     * @see #isGlobal()
     */
    public boolean isResource() {
        return !isGlobal();
    }

    /**
     * Indicates whether this permission applies to a given resource type.
     * <p>
     * Caveat: <em>global</em> permissions (that applies to all resources) will return {@code false}.
     *
     * @param resourceClass the type of the resource
     * @return {@code true} if this permission applies to instances of {@code resourceClass}
     * @see #isGlobal()
     * @see #isResource()
     */
    public boolean isResource(Class<?> resourceClass) {
        return !isGlobal() && Iterables.any(resourceTypes, c -> c.isAssignableFrom(resourceClass));
    }

    private void appendInheritedPermissions(ImmutableSet.Builder<Permission> builder, Permission p) {
        builder.add(p);
        for (Permission perm : p.getInheritedPermissions()) {
            appendInheritedPermissions(builder, perm);
        }
    }

    /**
     * Common i18n args used by permissions. They can't exist directly as constants in the {@code Permission} class
     * due to the forward-reference issue.
     */
    private static class I18nArgs {

        static final Object[] EMPTY = new Object[0];
        static final Object[] PRODUCT = new Object[] {
                Product.NAME
        };
    }

}
