package com.atlassian.bitbucket.log;

import javax.annotation.Nonnull;

/**
 * Service that allows runtime configuration of the logging subsystem.
 */
public interface LoggingService {

    /**
     * Retrieves the effective logging level of the specified logger. The effective logging level is level that is
     * defined for the specified logger or, if no level is defined for the specified logger, the effective level for
     * its parent logger.
     *
     * @param loggerName the name of the logger to retrieve the output level for
     * @return the effective logging level of the specified logger
     */
    @Nonnull
    String getLevel(String loggerName);

    /**
     * @return the effective logging level for the root logger, which is the ultimate parent of all loggers in the
     *         system
     */
    @Nonnull
    String getRootLevel();

    /**
     * Sets the level for the specified logger to the provided level. Valid levels are:
     * <ul>
     *     <li>OFF</li>
     *     <li>TRACE</li>
     *     <li>DEBUG</li>
     *     <li>INFO</li>
     *     <li>WARN</li>
     *     <li>ERROR</li>
     * </ul>
     * The special level {@code OFF} disables all output from the specified logger. All other levels include output at
     * that specific level as well as all output from the levels below them (meaning {@code TRACE} will include output
     * at {@code DEBUG}, {@code INFO}, {@code WARN} and {@code ERROR} levels).
     *
     * @param loggerName the name of the logger to adjust the output level for
     * @param level      the new output level for the logger
     */
    void setLevel(String loggerName, String level);

    /**
     * Sets the output level for the root logger, which is a special logger that is the ultimate parent of all other
     * loggers in the system.
     *
     * @param level the new output level for the root logger
     * @see #setLevel(String, String)
     */
    void setRootLevel(String level);
}
