package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.hook.ScmHookDetails;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;

/**
 * Request that is provided to {@code repository-hook} modules  for inspection
 * ({@link RepositoryHookService#preUpdate(RepositoryHookRequest) pre-hooks} and
 * {@link RepositoryHookService#postUpdate(RepositoryHookRequest) post-hooks}). The request describes the
 * {@link #getRefChanges() ref changes} that are about to be made (pre-hooks), or that have just been made
 * (post-hooks).
 * <p>
 * A request is marked as a {@link #isDryRun() dry-run} request when the proposed {@link #getRefChanges() ref changes}
 * need to be <em>quickly</em> verified, for instance to determine whether a button in the web UI should be enabled.
 * Hooks that are called with a {@link #isDryRun() dry-run} request should avoid expensive checks and operations.
 *
 * @since 5.0
 */
public interface RepositoryHookRequest {

    /**
     * Retrieves any additional context that was supplied to the request. This context is intended to
     * allow plugin developers to implement their own behavior around ref-changes. As a result, its contents
     * are <i>unspecified</i>; they depend on the installed plugins.
     * <p>
     * On {@link #isDryRun dry runs}, the context will <i>always</i> be empty.
     *
     * @return additional context provided when the ref-change was requested, if any
     */
    @Nonnull
    Map<String, Object> getContext();

    /**
     * @return the proposed ref changes. Can be {@link Collection#isEmpty() empty} for some dry-run requests where the
     *         target hash is not yet known because the relevant commit hasn't been created yet. As an example, this is
     *         the case for {@link MergeHookRequest merge requests}. In these cases, the specialized request type should
     *         provide sufficient information about the proposed change. For {@link MergeHookRequest}, this would be
     *         {@link MergeHookRequest#getFromRef()} and {@link MergeHookRequest#getToRef()}.
     */
    @Nonnull
    Collection<RefChange> getRefChanges();

    /**
     * @return the repository that's being updated
     */
    @Nonnull
    Repository getRepository();

    /**
     * @return SCM level hook details, if available. This will only be available when the hook is called synchronously
     *         as part of an SCM hook (such as pre-receive).
     */
    @Nonnull
    Optional<ScmHookDetails> getScmHookDetails();

    /**
     * @return the trigger of the request. This allows hook to adapt their logic based on the trigger.
     */
    @Nonnull
    RepositoryHookTrigger getTrigger();

    /**
     * @return {@code true} if this hook is a dry-run request, otherwise {@code false}
     */
    boolean isDryRun();
}
