package com.atlassian.bitbucket.event.server;

import com.atlassian.bitbucket.event.ApplicationEvent;
import com.atlassian.bitbucket.event.annotation.TransactionAware;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Event that is published when an application property is changed through ApplicationPropertiesService.
 * <p>
 * This event is internally audited with a HIGH priority.
 */
@TransactionAware
public abstract class ApplicationConfigurationChangedEvent<T> extends ApplicationEvent {

    private final Property property;
    private final T oldValue;
    private final T newValue;

    /**
     * Enum that lists the types of properties that can be changed. This enum is related to ApplicationProperty.Key,
     * but this is part of the public api and is based around logical units, whereas ApplicationProperty.Key
     * lists the individual config keys. For instance, there are 4 config keys for mail host config, but it
     * is only one logical configuration item.
     */
    public enum Property {

        BASE_URL,
        DISPLAY_NAME,
        HTTP_SCM_HOSTING,
        MAIL_HOST_CONFIGURATION,
        SERVER_EMAIL_ADDRESS,
        SERVER_IS_SETUP,
        TIME_ZONE
    }

    public ApplicationConfigurationChangedEvent(@Nonnull Object source, @Nonnull Property property,
                                                @Nullable T oldValue, @Nullable T newValue) {
        super(source);
        this.property = property;
        this.oldValue = oldValue;
        this.newValue = newValue;
    }

    @Nonnull
    public Property getProperty() {
        return property;
    }

    @Nullable
    public T getOldValue() {
        return oldValue;
    }

    @Nullable
    public T getNewValue() {
        return newValue;
    }
}
