package com.atlassian.bitbucket.event.repository;

import com.atlassian.bitbucket.repository.Repository;
import org.apache.commons.lang3.ObjectUtils;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Base class for constructing events which relate to repository modifications.
 * <p>
 * {@link #getRepository()} always returns the <i>current persisted state</i> for the repository. Whether that matches
 * the {@link #getOldValue() old values} or the {@link #getNewValue() new values} depends on the specific event, and
 * will be documented on the derived event class.
 */
@SuppressWarnings("unused") //Unused methods are for convenience, and may be used by plugins
public class AbstractRepositoryModificationEvent extends RepositoryEvent {

    protected final Repository newValue;
    protected final Repository oldValue;

    public AbstractRepositoryModificationEvent(@Nonnull Object source, @Nonnull Repository repository,
                                               @Nonnull Repository oldValue, @Nonnull Repository newValue) {
        super(source, repository);

        this.newValue = checkNotNull(newValue, "newValue");
        this.oldValue = checkNotNull(oldValue, "oldValue");
    }

    /**
     * Retrieves the <i>updated</i> repository values.
     *
     * @return the updated repository
     */
    @Nonnull
    public Repository getNewValue() {
        return newValue;
    }

    /**
     * Retrieves the <i>original</i> repository values.
     *
     * @return the original repository
     */
    @Nonnull
    public Repository getOldValue() {
        return oldValue;
    }

    public boolean isNameChanged() {
        return ObjectUtils.notEqual(getOldValue().getName(), getNewValue().getName());
    }

    public boolean isOriginChanged() {
        return ObjectUtils.notEqual(getOldValue().getOrigin(), getNewValue().getOrigin());
    }

    /**
     * @since 5.4
     */
    public boolean isMoved() {
        return ObjectUtils.notEqual(getOldValue().getProject().getId(), getNewValue().getProject().getId());
    }

    public boolean isSlugChanged() {
        return ObjectUtils.notEqual(getOldValue().getSlug(), getNewValue().getSlug());
    }

    public boolean isStateChanged() {
        return ObjectUtils.notEqual(getOldValue().getState(), getNewValue().getState());
    }
}
