package com.atlassian.bitbucket.event.pull;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;
import com.atlassian.bitbucket.pull.RescopeDetails;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Event that is raised when the ref for the source-branch and/or the target-branch of a pull request is updated. Note
 * that this does not necessarily mean that the list of commits that the pull request covers will change.
 */
@AsynchronousPreferred
@EventName("stash.pullrequest.rescoped")
public class PullRequestRescopedEvent extends PullRequestEvent {

    private final RescopeDetails addedCommits;
    private final String previousFromHash;
    private final String previousToHash;
    private final boolean fromHashUpdated;
    private final boolean toHashUpdated;
    private final RescopeDetails removedCommits;

    public PullRequestRescopedEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                    @Nonnull String previousFromHash, @Nonnull String previousToHash) {
        this(source, pullRequest, previousFromHash, previousToHash, null, null);
    }

    /**
     * @since 4.5
     */
    public PullRequestRescopedEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                    @Nonnull String previousFromHash, @Nonnull String previousToHash,
                                    @Nullable RescopeDetails addedCommits, @Nullable RescopeDetails removedCommits) {

        super(source, pullRequest, PullRequestAction.RESCOPED);

        this.previousFromHash = requireNonNull(previousFromHash, "previousFromHash");
        this.previousToHash = requireNonNull(previousToHash, "previousToHash");

        this.addedCommits = addedCommits;
        this.removedCommits = removedCommits;

        this.fromHashUpdated = !Objects.equals(pullRequest.getFromRef().getLatestCommit(), previousFromHash);
        this.toHashUpdated = !Objects.equals(pullRequest.getToRef().getLatestCommit(), previousToHash);
    }

    /**
     * @return details about what commits, if any were added to the scope of the pull request. The
     *         {@link RescopeDetails#getCommits() commits list} is limited for performance reasons. Use
     *         {@link RescopeDetails#getTotal()} ()} to determine the total number of commits that were added to the
     *         scope of the pull request. If more commits were added to the scope than are returned in this list, the
     *         list will contain the most recent commits that were added. Returns {@code null} if the scope change
     *         hasn't been determined.
     * @since 4.5
     */
    @Nullable
    public RescopeDetails getAddedCommits() {
        return addedCommits;
    }

    @Nonnull
    public String getPreviousFromHash() {
        return previousFromHash;
    }

    @Nonnull
    public String getPreviousToHash() {
        return previousToHash;
    }

    /**
     * @return details about what commits, if any were removed from the scope of the pull request. The
     *         {@link RescopeDetails#getCommits() commits list} is limited for performance reasons. Use
     *         {@link RescopeDetails#getTotal()} ()} to determine the total number of commits that were removed from the
     *         scope of the pull request. If more commits were removed from the scope than are returned in this list,
     *         the list will contain the most recent commits that were removed. Returns {@code null} if the scope change
     *         hasn't been determined.
     * @since 4.5
     */
    @Nullable
    public RescopeDetails getRemovedCommits() {
        return removedCommits;
    }

    /**
     * @return {@code true} if the from-hash has been updated in this scope change. {@code false} otherwise
     * @since 4.5
     */
    public boolean isFromHashUpdated() {
        return fromHashUpdated;
    }

    /**
     * @return {@code true} if the to-hash has been updated in this scope change. {@code false} otherwise
     * @since 4.5
     */
    public boolean isToHashUpdated() {
        return toHashUpdated;
    }
}
