package com.atlassian.bitbucket.event.branch;

import com.atlassian.bitbucket.hook.repository.RepositoryHookRequest;
import com.atlassian.bitbucket.hook.repository.StandardRepositoryHookTrigger;
import com.atlassian.bitbucket.repository.*;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Collections;

/**
 * A {@link RepositoryHookRequest hook request} that is raised just before a branch is created using the UI or REST.
 * Branches created by pushing will trigger a standard {@link RepositoryHookRequest} and not this specific type.
 *
 * @see StandardRepositoryHookTrigger#BRANCH_CREATE
 * @since 5.0
 */
public class BranchCreationHookRequest extends BranchHookRequest {

    private final Collection<RefChange> changes;

    private BranchCreationHookRequest(Builder builder) {
        super(builder);

        changes = Collections.singleton(new SimpleRefChange.Builder()
                .fromHash(StringUtils.repeat("0", 40))
                .to(getBranch())
                .type(RefChangeType.ADD)
                .build());
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return changes;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        public Builder(@Nonnull Repository repository, @Nonnull Branch branch) {
            super(repository, StandardRepositoryHookTrigger.BRANCH_CREATE, branch);
        }

        @Nonnull
        public BranchCreationHookRequest build() {
            return new BranchCreationHookRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
