package com.atlassian.bitbucket.dashboard;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import java.util.Date;

import static java.util.Objects.requireNonNull;

/**
 * A request for fetching pull request creation suggestions based on repository activity/history.
 * The purpose and behaviour of each field is described on its accessor.
 *
 * @since 4.10
 */
public class SuggestPullRequestsRequest {

    public static final int DEFAULT_LIMIT = 5;

    private final Date changesSince;
    private final int limit;
    private final ApplicationUser user;

    private SuggestPullRequestsRequest(Builder builder) {
        this.changesSince = builder.changesSince;
        this.limit = builder.limit;
        this.user = builder.user;
    }

    /**
     * All pull request suggestions are based on some form of activity/change that has occurred. This restricts
     * pull request suggestions to be based on activity up to a certain time/date in the past.
     */
    @Nonnull
    public Date getChangesSince() {
        return changesSince;
    }

    /**
     * Restricts the result set to return at most this many suggestions. If no limit was set by calling
     * {@link Builder#limit(int)} then the default {@link #DEFAULT_LIMIT} will be used.
     */
    public int getLimit() {
        return limit;
    }

    /**
     * Suggestions will be based on push activity for the given user.
     */
    @Nonnull
    public ApplicationUser getUser() {
        return user;
    }

    public static class Builder extends BuilderSupport {

        private final Date changesSince;
        private final ApplicationUser user;
        private int limit;

        /**
         * Constructs a builder for creating {@link SuggestPullRequestsRequest} objects.
         *
         * @param user          suggestions will be based on push activity for the given user
         * @param changesSince  all pull request suggestions are based on some form of activity/change that has
         *                      occurred. This restricts pull request suggestions to be based on activity up to
         *                      a certain time/date in the past
         */
        public Builder(@Nonnull ApplicationUser user, @Nonnull Date changesSince) {
            this.user = requireNonNull(user, "user");
            this.changesSince = requireNonNull(changesSince, "changesSince");
            limit = DEFAULT_LIMIT;
        }

        /**
         * Assembles a new {@link SuggestPullRequestsRequest} from the provided values.
         *
         * @return a pull request suggestion request instance
         */
        @Nonnull
        public SuggestPullRequestsRequest build() {
            return new SuggestPullRequestsRequest(this);
        }

        /**
         * Restricts the result set to return at most this many suggestions. If this is never called,
         * then the default {@link #DEFAULT_LIMIT} will be used.
         */
        @Nonnull
        public Builder limit(int value) {
            this.limit = value;
            return self();
        }

        @Nonnull
        protected Builder self() {
            return this;
        }
    }
}
