package com.atlassian.bitbucket.content;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An enumeration of file types in a diff.
 */
public enum DiffFileType {

    /**
     * The source ('from') file of the diff.
     */
    FROM(0),
    /**
     * The destination ('to') file of the diff,
     */
    TO(1);

    private final int id;

    DiffFileType(int id) {
        this.id = id;
    }

    /**
     * Returns the default file type for a given {@code segmentType}.
     *
     * @param segmentType segment type to match
     * @return the default matching file type for the provided segment type
     * @throws IllegalArgumentException if there is no file type currently associated with {@code segmentType}.
     */
    @Nonnull
    public static DiffFileType forSegmentType(@Nonnull DiffSegmentType segmentType) {
        checkNotNull(segmentType, "segmentType");
        if (segmentType == DiffSegmentType.REMOVED || segmentType == DiffSegmentType.CONTEXT) {
            return FROM;
        }
        if (segmentType == DiffSegmentType.ADDED) {
            return TO;
        }
        throw new IllegalArgumentException("Unrecognized segment type: " + segmentType);
    }

    @Nonnull
    public static DiffFileType fromId(int id) {
        for (DiffFileType value : values()) {
            if (value.getId() == id) {
                return value;
            }
        }
        throw new IllegalArgumentException("No DiffFileType is associated with ID [" + id + "]");
    }

    public int getId() {
        return id;
    }
}
