package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.comment.CommentThread;
import com.atlassian.bitbucket.comment.CommentThreadDiffAnchor;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.util.stream.Stream;

/**
 * A callback for receiving streaming diff details.
 * <p>
 * Diff contents are broken into 3 levels:
 * <ul>
 * <li><b>Segment</b> - A collection of one or more content lines sharing the same {@link DiffSegmentType}</li>
 * <li><b>Hunk</b> - A collection of segments, pinned to starting lines within the source and destination, representing
 * contiguous lines within the files being compared</li>
 * <li><b>Diff</b> - A collection of hunks comprising the changes between a given source and destination</li>
 * </ul>
 * In the case of deleted files, the segment will consistent entirely of {@link DiffSegmentType#REMOVED removed} lines;
 * conversely, for newly-added files, the segment will consist entirely of {@link DiffSegmentType#ADDED added} lines.
 * <p>
 * Certain {@link ChangeType types of changes}, such as a copy or a rename, may emit a diff without any hunks.
 * <p>
 * Note: Implementors are <i>strongly</i> encouraged to extend from {@link AbstractDiffContentCallback}. This interface
 * <i>will</i> change, over time, and any class implementing it directly will be broken by such changes. Extending from
 * the abstract class will help prevent such breakages.
 *
 * @see AbstractDiffContentCallback
 */
public interface DiffContentCallback {

    /**
     * Offers {@link CommentThread threads} for any comments which should be included in the diff to the callback.
     * Threads with both {@link CommentThreadDiffAnchor#isFileAnchor() File}
     * and {@link CommentThreadDiffAnchor#isLineAnchor() line} anchors may both be included in the provided stream.
     * <p>
     * Note: If this method is going to be invoked, it will <i>always</i> be invoked <i>before</i> the first invocation
     * of {@link #onDiffStart(Path, Path)}. This method will be called <i>at most once</i>. If multiple diffs are going
     * to be output to the callback, the {@link CommentThreadDiffAnchor#getPath() paths} of the anchors may reference
     * any of the files whose diffs will follow. Reconciling anchors to diffs is left to the implementation.
     *
     * @param threads a stream of zero or more threads describing comments within the diff or diffs that will be
     *                streamed to the callback. It can only be consumed once
     * @since 5.0
     */
    void offerThreads(@Nonnull Stream<CommentThread> threads) throws IOException;

    /**
     * Called to indicate a binary file differs. The exact differences cannot be effectively conveyed, however, so
     * no hunks/segments will be provided for binary files.
     *
     * @param src the source binary file
     * @param dst the destination binary file
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onBinary(@Nullable Path src, @Nullable Path dst) throws IOException;

    /**
     * Called upon reaching the end of the current overall diff, indicating no more changes exist for the current
     * source/destination pair.
     *
     * @param truncated {@code true} if any segment or hunk in the diff had to be truncated; otherwise, {@code false}
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onDiffEnd(boolean truncated) throws IOException;

    /**
     * Called to mark the start of an overall diff. The source and destination paths being compared, relative to their
     * containing {@link Repository repository}, are provided.
     * <ul>
     * <li>For {@link ChangeType#ADD added} files, the {@code src} path will be {@code null}</li>
     * <li>For {@link ChangeType#DELETE deleted} files, the {@code dst} path will be {@code null}</li>
     * <li>For all other {@link ChangeType changes}, both paths will be provided</li>
     * </ul>
     * 
     * @param src the source file being compared
     * @param dst the destination file being compared
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onDiffStart(@Nullable Path src, @Nullable Path dst) throws IOException;

    /**
     * Called after the final {@link #onDiffEnd(boolean)}, after all diffs have been streamed.
     *
     * @param summary summarizes the request and the streamed diff
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onEnd(@Nonnull DiffSummary summary) throws IOException;

    /**
     * Called upon reaching the end of a hunk of segments within the current overall diff.
     *
     * @param truncated {@code true} if any segment in the hunk had to be truncated; otherwise, {@code false}
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onHunkEnd(boolean truncated) throws IOException;

    /**
     * Called to mark the start of a new hunk within the current overall diff, containing one or more contiguous
     * segments of lines anchored at the provided line numbers in the source and destination files.
     *
     * @param srcLine the line in the source file at which the hunk starts
     * @param srcSpan the number of lines spanned by this hunk in the source file
     * @param dstLine the line in the destination file at which the hunk starts
     * @param dstSpan the number of lines spanned by this hunk in the destination file
     * @throws IOException May be thrown by implementations which perform I/O.
     * @deprecated in 5.5 for removal in 6.0. Callbacks should implement
     *             {@link #onHunkStart(int, int, int, int, String)} instead.
     */
    @Deprecated
    @SuppressWarnings("DeprecatedIsStillUsed") //The only remaining usages are below and in AbstractDiffContentCallback
    default void onHunkStart(int srcLine, int srcSpan, int dstLine, int dstSpan) throws IOException {
    }

    /**
     * Called to mark the start of a new hunk within the current overall diff, containing one or more contiguous
     * segments of lines anchored at the provided line numbers in the source and destination files.
     *
     * @param srcLine the line in the source file at which the hunk starts
     * @param srcSpan the number of lines spanned by this hunk in the source file
     * @param dstLine the line in the destination file at which the hunk starts
     * @param dstSpan the number of lines spanned by this hunk in the destination file
     * @param context an optional context, such as a method or class name, for the hunk's lines
     * @throws IOException May be thrown by implementations which perform I/O.
     * @since 5.5
     */
    @SuppressWarnings("deprecation")
    default void onHunkStart(int srcLine, int srcSpan, int dstLine, int dstSpan, @Nullable String context)
            throws IOException {
        onHunkStart(srcLine, srcSpan, dstLine, dstSpan);
    }

    /**
     * Called upon reaching the end of a segment within the current hunk, where a segment may end either because lines
     * with a different type were encountered or because there are no more contiguous lines in the hunk.
     * <p>
     * Note: Internal restrictions may prevent streaming full segments if they are overly large. For example, if a new
     * file containing tens of thousands of lines is added (or an existing file of such size is deleted), the resulting
     * segment may be truncated.
     *
     * @param truncated {@code true} if the overall segment exceeded the maximum allowed lines, indicating one or more
     *                  lines were omitted from the segment; otherwise {@code false} to indicate all lines were sent
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onSegmentEnd(boolean truncated) throws IOException;

    /**
     * Called to process a line within the current segment.
     * <p>
     * {@link PullRequest Pull request} diffs may contain conflicts, if the pull request
     * cannot be merged cleanly. The {@code marker} conveys this information.
     * <ul>
     *     <li>{@code null}: The line is not conflicted</li>
     *     <li>{@link ConflictMarker#MARKER}: The line is a conflict marker</li>
     *     <li>{@link ConflictMarker#OURS}: The line is conflicting, and is present in the merge target</li>
     *     <li>{@link ConflictMarker#THEIRS}: The line is conflicting, and is present in the merge source</li>
     * </ul>
     * <p>
     * Note: Internal restrictions may prevent receiving the full line if it contains too many characters. When this
     * happens {@code truncated} will be set to {@code true} to indicate the line is not complete.
     *
     * @param line      a single line of content
     * @param marker    indicates whether the line is conflicted and, if it is, which side of the conflict it's on
     * @param truncated {@code true} if the line exceeded the maximum allowed length and was truncated; otherwise,
     *                  {@code false} to indicate the line is complete
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onSegmentLine(@Nonnull String line, @Nullable ConflictMarker marker, boolean truncated) throws IOException;

    /**
     * Called to mark the start of a new segment within the current hunk, containing one or more contiguous lines which
     * all share the same {@link DiffSegmentType type}.
     *
     * @param type the shared type for all lines in the new segment
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onSegmentStart(@Nonnull DiffSegmentType type) throws IOException;

    /**
     * Called before the first {@link #onDiffStart(Path, Path)}.
     *
     * @param context provides details about the diff request for which results are being streamed
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    void onStart(@Nonnull DiffContext context) throws IOException;
}
