package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.user.Person;

import javax.annotation.Nonnull;
import java.util.Date;

/**
 * Provides blame (or praise) data for a span of lines within a file.
 */
public interface Blame {

    /**
     * The author who last updated the lines in the span.
     *
     * @return the author
     */
    @Nonnull
    Person getAuthor();

    /**
     * The timestamp at which the author committed the revision.
     *
     * @return the author's commit timestamp
     */
    @Nonnull
    Date getAuthorTimestamp();

    /**
     * The committer who last updated the lines in the span.
     *
     * @return the committer
     * @since 5.0
     */
    @Nonnull
    default Person getCommitter() {
        return getAuthor();
    }

    /**
     * The timestamp at which the committer created the revision.
     *
     * @return the committer's timestamp
     * @since 5.0
     */
    @Nonnull
    default Date getCommitterTimestamp() {
        return getAuthorTimestamp();
    }

    /**
     * The full commit hash for the revision.
     *
     * @return the commit hash
     */
    @Nonnull
    String getCommitId();

    /**
     * A shorted version of the commit hash for the revision, suitable for display.
     *
     * @return the display commit hash
     */
    @Nonnull
    String getCommitDisplayId();

    /**
     * The name the file had when the lines in the span were updated, which may not match the file's current name.
     *
     * @return the filename when the revision occurred
     */
    @Nonnull
    String getFileName();

    /**
     * The <i>1-based</i> line number in the file at which this span of blame begins.
     *
     * @return the line number beginning this span of blame
     */
    int getLineNumber();

    /**
     * The number of lines in the span, to which the same blame information (author, commit hash, etc.) applies. This
     * value will never be negative, but may be {@code 0} for an empty file.
     *
     * @return the number of lines covered by this span of blame information
     */
    int getSpannedLines();
}
