package com.atlassian.bitbucket.concurrent;

import javax.annotation.Nonnull;
import java.io.Serializable;
import java.util.function.Function;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Configuration object used for {@link ConcurrencyService#getBucketedExecutor(String, BucketedExecutorSettings)
 * constructing} {@link BucketedExecutor} instances.
 *
 * @param <T> type task type
 */
public class BucketedExecutorSettings<T extends Serializable> {

    private final int batchSize;
    private final Function<T, String> bucketIdExtractor;
    private final ConcurrencyPolicy concurrencyPolicy;
    private final int maxAttempts;
    private final int maxConcurrency;
    private final BucketProcessor<T> processor;

    private BucketedExecutorSettings(Builder<T> builder) {
        this.batchSize = builder.batchSize;
        this.bucketIdExtractor = builder.bucketIdExtractor;
        this.concurrencyPolicy = builder.concurrencyPolicy;
        this.maxAttempts = builder.maxAttempts;
        this.maxConcurrency = builder.maxConcurrency;
        this.processor = builder.processor;
    }

    /**
     * @return the number of items from a bucket that should be processed in a single {@link BucketProcessor} run. The
     *         number of items that are passed to a {@link BucketProcessor} is guaranteed to be &lt;= the configured
     *         batch size.
     */
    public int getBatchSize() {
        return batchSize;
    }

    /**
     * @return the function used to group items in buckets
     */
    @Nonnull
    public Function<T, String> getBucketIdExtractor() {
        return bucketIdExtractor;
    }

    /**
     * @return the maximum number of times processing of a bucket should be attempted
     */
    public int getMaxAttempts() {
        return maxAttempts;
    }

    /**
     * @return the maximum number of different buckets that can be concurrently processed
     */
    public int getMaxConcurrency() {
        return maxConcurrency;
    }

    /**
     * @return the concurrency policy. If the concurrency policy is {@link ConcurrencyPolicy#PER_CLUSTER}, the {@link
     *         #getMaxConcurrency() maximum concurrency} is translated into a per-node maximum by dividing the maximum
     *         by the number of nodes in the cluster (rounded up).
     */
    @Nonnull
    public ConcurrencyPolicy getMaxConcurrencyPolicy() {
        return concurrencyPolicy;
    }

    /**
     * @return the processor responsible for processing buckets of tasks
     */
    @Nonnull
    public BucketProcessor<T> getProcessor() {
        return processor;
    }

    /**
     * Builder for {@link BucketedExecutorSettings}
     * @param <T> the task type
     */
    public static class Builder<T extends Serializable> {

        /**
         * @deprecated this field will no longer be public in 6.0
         */
        @Deprecated
        public int maxConcurrency;

        private final Function<T, String> bucketIdExtractor;
        private final BucketProcessor<T> processor;

        private int batchSize;
        private ConcurrencyPolicy concurrencyPolicy;
        private int maxAttempts;

        /**
         * @param bucketIdExtractor the function used to group items in buckets
         * @param processor the processor responsible for processing buckets of tasks
         */
        public Builder(@Nonnull Function<T, String> bucketIdExtractor, @Nonnull BucketProcessor<T> processor) {
            this.bucketIdExtractor = checkNotNull(bucketIdExtractor, "bucketIdExtractor");
            this.processor = checkNotNull(processor, "processor");

            batchSize = Integer.MAX_VALUE;
            maxAttempts = 2;
            concurrencyPolicy = ConcurrencyPolicy.PER_NODE;
            maxConcurrency = 2;
        }

        /**
         * @param value the number of items from a bucket that should be processed in a single {@link BucketProcessor}
         *              run. The number of items that are passed to a {@link BucketProcessor} is guaranteed to be &lt;=
         *              the configured batch size.
         * @return the builder
         */
        public Builder<T> batchSize(int value) {
            batchSize = value;
            return this;
        }

        public BucketedExecutorSettings<T> build() {
            return new BucketedExecutorSettings<>(this);
        }

        /**
         * @param value the maximum number of times processing of a bucket should be attempted
         * @return the builder
         */
        public Builder<T> maxAttempts(int value) {
            maxAttempts = value;
            return this;
        }

        /**
         * @param value the maximum number of different buckets that can be concurrently processed. In a clustered
         *              instance, this maximum is translated into a per-node maximum by dividing the maximum by the
         *              number of nodes in the cluster (rounded up).
         * @return the builder
         */
        public Builder<T> maxConcurrency(int value, ConcurrencyPolicy policy) {
            concurrencyPolicy = policy;
            maxConcurrency = value;
            return this;
        }
    }
}
