package com.atlassian.bitbucket.compare;

import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A request to compare two refs or commits.
 *
 * @see CompareRef
 */
public class CompareRequest {

    private final CompareRef fromRef;
    private final CompareRef toRef;

    private CompareRequest(@Nonnull Builder builder) {
        this.fromRef = builder.fromRef;
        this.toRef = builder.toRef;
    }

    /**
     * @return the {@link CompareRef ref} serving as the base of the changes to compare
     */
    @Nonnull
    public CompareRef getFromRef() {
        return fromRef;
    }

    /**
     * @return the {@link CompareRef ref} serving as the target of those changes
     */
    @Nonnull
    public CompareRef getToRef() {
        return toRef;
    }

    public static class Builder {

        private CompareRef fromRef;
        private CompareRef toRef;

        public Builder() {
        }

        public Builder(@Nonnull CompareDiffRequest request) {
            fromRef = checkNotNull(request, "request").getFromRef();
            toRef = request.getToRef();
        }

        /**
         * Convenience constructor for creating a {@code Builder} using the refs from a provided {@link PullRequest}.
         *
         * @param pullRequest the pull request to build a {@link CompareRequest} from
         */
        public Builder(@Nonnull PullRequest pullRequest) {
            fromRef = new CompareRef(checkNotNull(pullRequest, "pullRequest").getFromRef());
            toRef = new CompareRef(pullRequest.getToRef());
        }

        @Nonnull
        public CompareRequest build() {
            checkArgument(fromRef != null && toRef != null, "both a source and target ref are required");
            return new CompareRequest(this);
        }

        @Nonnull
        public Builder fromRef(@Nonnull CompareRef fromRef) {
            this.fromRef = checkNotNull(fromRef, "fromRef");
            return this;
        }

        @Nonnull
        public Builder fromRef(@Nonnull String id, @Nonnull Repository repository) {
            return fromRef(new CompareRef(id, repository));
        }

        @Nonnull
        public Builder toRef(@Nonnull CompareRef toRef) {
            this.toRef = checkNotNull(toRef, "toRef");
            return this;
        }

        @Nonnull
        public Builder toRef(@Nonnull String id, @Nonnull Repository repository) {
            return toRef(new CompareRef(id, repository));
        }
    }

}
