package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.bulk.BulkCommitCallback;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSetMultimap;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.OptionalInt;
import java.util.Set;

import static java.util.Objects.requireNonNull;
import static java.util.OptionalInt.empty;
import static java.util.OptionalInt.of;

/**
 * Defines a request to retrieve any arbitrary collection of commits that are in any number of repositories.
 *
 * @see CommitService#streamCommits(BulkCommitsRequest, BulkCommitCallback)
 * @since 5.8
 */
public class BulkCommitsRequest {

    private static final int UNLIMITED_MESSAGE_LENGTH = -1;

    private final Map<Repository, Set<String>> commits;
    private final boolean ignoringMissing;
    private final int maxMessageLength;

    @SuppressWarnings("unchecked")
    private BulkCommitsRequest(Builder builder) {
        commits = (Map) builder.commits.build().asMap();
        ignoringMissing = builder.ignoringMissing;
        maxMessageLength = builder.maxMessageLength;
    }

    /**
     * @return the collection of commits to retrieve, mapped by the repository to search in
     */
    @Nonnull
    public Map<Repository, Set<String>> getCommits() {
        return commits;
    }

    /**
     * @return the max message length to be processed for any of the commits, which may be {@code empty()} (the default)
     *         to request the full commit message regardless of length or {@code of(0)} to omit messages entirely
     */
    @Nonnull
    public OptionalInt getMaxMessageLength() {
        return maxMessageLength == UNLIMITED_MESSAGE_LENGTH ? empty() : of(maxMessageLength);
    }

    /**
     * @return {@code true} if missing requested commits should be ignored; otherwise, {@code false} if they should
     *         trigger a {@link NoSuchCommitException} (the default)
     */
    public boolean isIgnoringMissing() {
        return ignoringMissing;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSetMultimap.Builder<Repository, String> commits;

        private boolean ignoringMissing;
        private int maxMessageLength;

        public Builder() {
            commits = ImmutableSetMultimap.builder();
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
        }

        @Nonnull
        public BulkCommitsRequest build() {
            return new BulkCommitsRequest(this);
        }

        @Nonnull
        public Builder commit(@Nonnull MinimalRepositoryCommit value) {
            commits.put(requireNonNull(value, "commit").getRepository(), value.getId());

            return this;
        }

        @Nonnull
        public Builder commit(@Nonnull Repository repository, @Nonnull String commit) {
            commits.put(requireNonNull(repository, "repository"), requireNonBlank(commit, "commit"));

            return this;
        }

        @Nonnull
        public Builder commits(@Nonnull Iterable<MinimalRepositoryCommit> values) {
            requireNonNull(values, "commits").forEach(this::commit);

            return this;
        }

        @Nonnull
        public Builder commits(@Nonnull Repository repository, @Nonnull Iterable<String> value) {
            commits.putAll(requireNonNull(repository, "repository"), requireNonNull(value, "commits"));

            return this;
        }

        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoringMissing = value;

            return this;
        }

        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return this;
        }
    }
}
