define('hipchat/feature/repo-room-table', [
    'aui',
    'hipchat/feature/repo-room-table/analytics',
    'jquery',
    'lodash',
    'bitbucket/util/navbuilder',
    'bitbucket/util/server',
    'bitbucket/internal/widget/confirm-dialog',
    'bitbucket/internal/widget/paged-table'
], function (
    AJS,
    analytics,
    $,
    _,
    navbuilder,
    server,
    ConfirmDialog,
    PagedTable
) {
    'use strict';

    var AVAILABLE_ROOM_STATE = 'AVAILABLE';

    /**
     * Creates a RepoRoomTable
     *
     * @param {Object} options
     * @param {navbuilder.Builder} options.baseUrlBuilder - Navbuilder for that base URL to load more table rows.
     * @param {Object} options.categories - Map containing category information split into different categories
     * @constructor
     * @extends PagedTable
     */
    function RepoRoomTable (options) {
        var defaultOptions = {
            target: "#hipchat-repo-room-table",
            allFetchedMessageHtml: bitbucket.internal.hipchat.feature.repoRoomTable.allMessagesFetched(),
            rowSelector: '> tbody',
            targetedItem: '> tbody',
            paginationContext: 'hipchat-repo-room',
            showProject: false
        };

        options = _.extend({}, defaultOptions, options);

        PagedTable.call(this, options);

        _.bindAll(this, 'addRoomMapping', 'saveRoomMapping', 'onEditNotification', 'onCloseEditNotification', 'onNotificationChanged');
        this.categories = options.categories;
        this.baseUrlBuilder = options.baseUrlBuilder;

        this.$table = $('#hipchat-repo-room-table');

        this.initMappingTable();
        this.initRemoveDialog();

        this.$table.on('change', '.notification-item', this.onNotificationChanged);
    }

    _.extend(RepoRoomTable.prototype, PagedTable.prototype);

    /**
     * Inits the table and binds all click handlers
     */
    RepoRoomTable.prototype.initMappingTable = function () {
        this.$table.on('click', '.edit-notification', this.onEditNotification);
        this.$table.on('click', '.close-edit-notification', this.onCloseEditNotification);
        // Add a tipsy tooltip to tbodies - they may have status information titles for a room (e.g. deleted / private)
        this.$table.find('tbody').tooltip({
            delayIn: 0,
            gravity: 's'
        });
    };

    /**
     * Inits the remove dialog that is shown to confirm that the mapping should be removed.
     */
    RepoRoomTable.prototype.initRemoveDialog = function ( ) {
        var self = this;

        var deleteButtonSelector = '.trash-room-mapping';
        var $panelContent = $(bitbucket.internal.hipchat.feature.repoRoomTable.confirmDialogContent());

        var confirmDialog = new ConfirmDialog({
            id: "delete-mapping-dialog",
            titleText: AJS.I18n.getText('bitbucket.plugins.hipchat.remove-dialog.title'),
            titleClass: 'warning-header',
            confirmButtonClass: 'delete-confirm-button',
            panelContent: $panelContent,
            submitToHref: true,
            submitText: AJS.I18n.getText('bitbucket.plugins.hipchat.remove-dialog.button.stop')
        }, { type: 'DELETE' });

        confirmDialog.attachTo(deleteButtonSelector, function (triggerEl, dialog) {
            var $triggerEl = $(triggerEl);

            var mapping = extractRowMappingInfo($triggerEl);
            $triggerEl.attr('href', getRoomRestNavBuilder(mapping).build());

            var roomName = self.findMappingTbody(mapping).find('.room-name').text();
            dialog.getPanel(0).body.find('.display-name').text(roomName);
        });
        confirmDialog.addConfirmListener(function (promise, $triggerEl) {
            var mapping = extractRowMappingInfo($triggerEl);
            self.findMappingTbody(mapping).remove();
            analytics.mappingDeleted(mapping);
        });
    };

    /**
     * Creates a new mapping in the table and opens it for editing.
     *
     * The UI shows up optimistically before the REST request returns.
     *
     * @param {Room} room - The Room object returned from the rest endpoint.
     * @param {Repository} repo - A Repository object that the mapping show be created with.
     */
    RepoRoomTable.prototype.addRoomMapping = function (room, repo) {
        var mapping = {
            projectKey: repo.project.key,
            repoId: repo.id,
            repoSlug: repo.slug,
            roomId: room.id
        };
        if (this.hasRow(mapping)) {
            this.expandRow(mapping);
            return;
        }
        var $tbody = $(bitbucket.internal.hipchat.feature.repoRoomTable.renderRepoRoomMapping({
            repository: repo,
            categories: this.categories,
            activateDefaults: true,
            showProject: this.options.showProject,
            roomConfig: {
                roomDetails: {
                    roomName: room.name,
                    roomId: room.id,
                    state: AVAILABLE_ROOM_STATE
                }
            }
        }));
        $tbody.find('.notification-item').attr('checked', 'checked');
        $tbody.insertAfter(this.$table.find('thead'));

        this.expandRow(mapping);
        this.$table.find('thead.hidden').removeClass('hidden');
        this.saveRoomMapping(mapping);

        analytics.mappingAdded(mapping);
    };

    /**
     * Checks if a row mapping exists in the table.
     *
     * @param {RepoRoomMapping} mapping
     * @returns {boolean}
     */
    RepoRoomTable.prototype.hasRow = function (mapping) {
        return this.$table.find('> tbody[data-room-id=' + mapping.roomId + ']' +
            '[data-repo-slug="' + mapping.repoSlug + '"][data-project-key="' + mapping.projectKey + '"]').length !== 0;
    };

    /**
     * Builds a URL for loading the next page of the table
     * Implemented for PagedTable
     *
     * @param {number} start - The offset to start the page
     * @param {number} limit - The max number of rows the URL should return
     * @returns {string} the built URL
     */
    RepoRoomTable.prototype.buildUrl = function (start, limit) {
        return this.baseUrlBuilder
            .withParams({
                start: start,
                limit: limit
            }).build();
    };

    /**
     * Expands a row of the table to reveal its configuration options.
     * Will close all ofther expanded rows.
     *
     * @param {RepoRoomMapping} mapping - the mapping object to describe the row.
     */
    RepoRoomTable.prototype.expandRow = function (mapping) {
        this.$table.find('.expanded').removeClass('expanded');
        var $tbody = this.findMappingTbody(mapping);
        $tbody[0].scrollIntoView();
        $tbody.addClass('expanded');
    };

    /**
     * Find the <tbody> for the specified mapping.
     *
     * @param {RepoRoomMapping} mapping - the mapping object representing the row to find.
     * @returns {jQuery}
     */
    RepoRoomTable.prototype.findMappingTbody = function (mapping) {
        return this.$table.find('tbody[data-room-id="' + mapping.roomId + '"]' +
        '[data-project-key="' + mapping.projectKey + '"]' +
        '[data-repo-slug="' + mapping.repoSlug + '"]');
    };

    /**
     * Adds new rows to the table when they are loaded by the PagedTable.
     *
     * This implementation acts as though the attachmentMethod is always 'append'
     *
     * @param {Object} data - the data that came back from the server
     */
    RepoRoomTable.prototype.handleNewRows = function (data) {
        var self = this;
        data.values.forEach(function (repoConfig) {
            repoConfig.roomConfigurations.forEach(function (roomConfig) {
                var tbody = bitbucket.internal.hipchat.feature.repoRoomTable.renderRepoRoomMapping({
                    roomConfig: roomConfig,
                    categories: self.categories,
                    repository: repoConfig.repository,
                    showProject: self.options.showProject
                });
                self.$table.append(tbody);
            });
        });
    };

    /**
     * Bound to the close edit notifications button
     * Closes the edit notifications panel.
     *
     * @param {Event} e
     */
    RepoRoomTable.prototype.onCloseEditNotification = function (e) {
        $(e.target).closest('tbody').removeClass('expanded');
    };

    /**
     * Bound to the open edit notifications button.
     * Opens the edit notifications panel.
     *
     * @param {Event} e
     */
    RepoRoomTable.prototype.onEditNotification = function (e) {
        this.expandRow(extractRowMappingInfo(e.target));
    };

    /**
     * Updates the notification setting for one type of notification.
     *
     * @param {Event} e - The event on the checkbox element.
     */
    RepoRoomTable.prototype.onNotificationChanged = function (e) {
        var mapping = extractRowMappingInfo(e.target);
        var notificationId = $(e.target).attr('data-key');

        server.rest({
            method: e.target.checked ? 'PUT' : 'DELETE',
            url: getRoomRestNavBuilder(mapping)
                .addPathComponents('notifications', notificationId).build()
        });
        analytics.mappingsChanged(mapping, $(e.target).closest('tbody'));
    };

    /**
     * Adds/removes a room enabling or disabling all notification types at once.
     *
     * @param {RepoRoomMapping} mapping
     * @returns {Promise} A jQuery promise for the XHR request.
     */
    RepoRoomTable.prototype.saveRoomMapping = function (mapping) {
        return server.rest({
            method: 'PUT',
            url: getRoomRestNavBuilder(mapping).build()
        });
    };

    /**
     * Represents a Mapping between a Repository and a Room.
     * @typedef {Object} RepoRoomMapping
     * @property {string} projectKey - The project key that the repository is under.
     * @property {string} repoId - The ID of the repository.
     * @property {string} repoSlug - The slug of the repository.
     * @property {number} roomId - The rooms ID number.
     */

    /**
     *
     * @param {HTMLElement} el
     * @returns {RepoRoomMapping}
     */
    function extractRowMappingInfo (el) {
        return {
            projectKey: extractRowMapping(el, 'project-key'),
            repoId: extractRowMapping(el, 'repo-id'),
            repoSlug: extractRowMapping(el, 'repo-slug'),
            roomId: extractRowMapping(el, 'room-id')
        };
    }

    /**
     * Extracts some data information from the given element.
     *
     * It will search back up the hierarchy looking for a <tbody> with the information it needs on it
     * @param {HtmlElement} el - The element to start the lookup on
     * @param {string} dataKey - the data key to lookup
     * @returns {string}
     */
    function extractRowMapping(el, dataKey) {
        return $(el).closest('tbody').attr('data-' + dataKey);
    }

    /**
     * Builds a rest navbuilder for the specified room, in the current repo.
     *
     * @param {RepoRoomMapping} mapping
     * @returns {navBuilder.Builder}
     */
    function getRoomRestNavBuilder (mapping) {
        return navbuilder.rest('hipchat-integrations')
            .project(mapping.projectKey)
            .repo(mapping.repoSlug)
            .addPathComponents('rooms', mapping.roomId);
    }

    return RepoRoomTable;
});