package com.atlassian.beehive.db;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.core.ClusterLockStatus;
import com.atlassian.beehive.core.LockRegistry;
import com.atlassian.beehive.core.ManagedClusterLockService;
import com.atlassian.beehive.db.spi.ClusterLockDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;
import java.util.Collection;

/**
 * This is the database backed implementation of ClusterLockService that can be used by clustered applications.
 * <p>
 *     This implementation relies on the host application providing SPI implementations for Data Access objects.
 * </p>
 *
 *  Warning: Any alterations of tables used by DAO SPI implementations, if performed at runtime, need to be followed by calling {@link #resetDatabaseState()}.
 *  Failing to do so immediately after the changes, may result in unexpected locks behavior.
 *  Furthermore, the existing unreleased locks, that were held since before the db alteration, no logner guarantee clusterwide exclusiveness until they are successfully reacquired.
 */
public class DatabaseClusterLockService implements ManagedClusterLockService
{
    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLockService.class);
    private final ClusterLockDao clusterLockDao;
    private final LockRegistry<DatabaseClusterLock> registry = new LockRegistry<DatabaseClusterLock>()
    {
        @Override
        protected DatabaseClusterLock createLock(String lockName)
        {
            return new DatabaseClusterLock(lockName, clusterLockDao, databaseClusterLockLeaseRenewer);
        }
    };
    private final DatabaseClusterLockLeaseRenewer databaseClusterLockLeaseRenewer;


    public DatabaseClusterLockService(final ClusterLockDao clusterLockDao)
    {
        this.clusterLockDao = clusterLockDao;
        this.databaseClusterLockLeaseRenewer = new DatabaseClusterLockLeaseRenewer();
        clusterLockDao.releaseLocksHeldByNode();
    }


    @Override
    public Collection<DatabaseClusterLock> getAllKnownClusterLocks()
    {
        return registry.getAllKnownClusterLocks();
    }

    @Nonnull
    @Override
    public Collection<ClusterLockStatus> getStatusesOfAllHeldClusterLocks() {
        return clusterLockDao.getAllHeldClusterLocks();
    }

    @Override
    public ClusterLock getLockForName(@Nonnull String lockName)
    {
        return registry.getLockForName(lockName);
    }

    @PreDestroy
    public void shutdown() {
        databaseClusterLockLeaseRenewer.shutdown();
    }

    /**
     * Reset internal state of the locks to adjust to external changes in tables used by DAO at runtime.
     *
     */
    public void resetDatabaseState() {
        final Collection<DatabaseClusterLock> existingLocks = registry.getAllKnownClusterLocks();
        log.info("External changes in database detected, resetting {} known locks...", existingLocks.size());
        existingLocks.forEach(DatabaseClusterLock::resetDatabaseState);
    }
}

