package com.atlassian.beehive.db;

import com.atlassian.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

/**
 * A service that periodically renews cluster lock leases, as long as:
 * 1. Our local status says this node still owns the cluster lock
 * 2. No other node has stolen the lock in the meantime (e.g. when we haven't renewed the lease for long enough)
 * 3. The thread that owns the lock is still alive
 *
 */
class DatabaseClusterLockLeaseRenewer {

    private final ScheduledThreadPoolExecutor executor;
    private final ConcurrentHashMap<String, RenewalInstance> renewerMap;


    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLockLeaseRenewer.class);

    DatabaseClusterLockLeaseRenewer() {
        executor = new ScheduledThreadPoolExecutor(1, runnable -> {
            Thread t = new Thread(runnable, "cluster-lock-lease-renewal-thread");
            t.setDaemon(true);
            return t;
        });

        executor.setRemoveOnCancelPolicy(true);

        renewerMap = new ConcurrentHashMap<>();
    }

    void onLock(final DatabaseClusterLock lock) {
        log.trace("onLock: " + lock.getName());
        final RenewalTask task = createRenewalTask(lock);
        final ScheduledFuture future = executor.scheduleAtFixedRate(task, 5000, LockExpiryConfiguration.getRenewalIntervalInSeconds() * 1000, TimeUnit.MILLISECONDS);
        final RenewalInstance previous = renewerMap.put(lock.getName(), new RenewalInstance(task, future));
        if (previous != null) {
            log.warn("Detected previous, unfinished job when scheduling new renewer job for " + lock.getName()+", cancelling old job...");
            previous.getFuture().cancel(true);
        }
    }

    void onUnlock(final DatabaseClusterLock lock) {
        log.trace("onUnlock: " + lock.getName());
        unscheduleForLock(lock.getName());
    }

    void shutdown() {
        executor.shutdownNow();
    }

    @VisibleForTesting
    RenewalTask createRenewalTask(final DatabaseClusterLock lock) {
        return new RenewalTask(lock);
    }

    private void unscheduleForLock(final String lockName) {
        final RenewalInstance renewal = renewerMap.remove(lockName);
        if (renewal != null) {
            renewal.getFuture().cancel(true);
        }
    }

    private void unscheduleTask(RenewalTask task) {
        final Predicate<Map.Entry<String, RenewalInstance>> entryPredicate = e -> task.equals(e.getValue()
                .getTask());
        renewerMap.entrySet().stream()
                .filter(entryPredicate)
                .findFirst()
                .map(Map.Entry::getValue).map(RenewalInstance::getFuture)
                .ifPresent(f -> f.cancel(false));
        renewerMap.entrySet().removeIf(entryPredicate);
    }

    private static class RenewalInstance {

        private RenewalTask task;
        private ScheduledFuture future;

        public RenewalInstance(final RenewalTask task, final ScheduledFuture future) {
            this.task = task;
            this.future = future;
        }

        public RenewalTask getTask() {
            return task;
        }

        public ScheduledFuture getFuture() {
            return future;
        }
    }

    class RenewalTask implements Runnable {

        final private DatabaseClusterLock lock;

        private final Long startedAt = System.currentTimeMillis();

        public RenewalTask(DatabaseClusterLock lock) {
            this.lock = lock;
        }

        @Override
        public void run() {
            try {
                log.trace("Renewing lease on lock: " + getLockName());
                if (lock.renew()) {
                    log.trace("Successfully renewed lease on lock: {}, held for {} ms", getLockName(), System.currentTimeMillis() - startedAt);
                }
            } catch (DeadOwnerThreadException e) {
                log.warn("Detected that lock {} is owned by a dead thread. Canceling renewer process for this lock...", getLockName());
                unscheduleSelf();
            } catch (Throwable t) {
                log.error("Failed to renew lease on lock: , " + getLockName() + ". Error occured during attempt to renew this lock. This should never happen here. Will retry on next scheduled run...", t);
            }
        }

        private void unscheduleSelf() {
            unscheduleTask(this);
        }

        private String getLockName() {
            return lock.getName();
        }
    }
}
