package com.atlassian.beehive.db;

import com.atlassian.beehive.core.ClusterLockStatus;
import com.atlassian.beehive.core.ManagedClusterLock;
import com.atlassian.beehive.core.stats.StatisticsKey;
import com.atlassian.beehive.db.spi.ClusterLockDao;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.function.BooleanSupplier;

/**
 * Cluster Lock using the Database as its backing store.
 * For locks represented by instances of this class to be usable during runtime, it is expected that they and only they exclusively manage their database representations via {@link ClusterLockDao}
 */
class DatabaseClusterLock implements ManagedClusterLock
{

    private static final int INITIAL_SLEEP_MILLIS = 100;
    private static final int MAX_SLEEP_MILLIS = 10 * 1000;
    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLock.class);
    private static final int MAX_RETRIES = 3;

    private final String lockName;
    private final ClusterLockDao clusterLockDao;

    private final AtomicReference<Owner> ownerRef = new AtomicReference<Owner>();
    private final AtomicInteger depth = new AtomicInteger();

    private final StatisticsHolder stats = new StatisticsHolder();
    private final Supplier<ClusterLockStatus> databaseLockStatusSupplier;
    private final AtomicBoolean inserted = new AtomicBoolean(false);

    private final Object monitor=new Object();

    private DatabaseClusterLockLeaseRenewer databaseClusterLockLeaseRenewer;

    public DatabaseClusterLock(String lockName, ClusterLockDao clusterLockDao, DatabaseClusterLockLeaseRenewer databaseClusterLockLeaseRenewer)
    {
        this.lockName = lockName;
        this.clusterLockDao = clusterLockDao;
        this.databaseClusterLockLeaseRenewer = databaseClusterLockLeaseRenewer;
        this.databaseLockStatusSupplier = Suppliers.memoizeWithExpiration(() -> {
            final ClusterLockStatus clusterLockStatus = clusterLockDao.getClusterLockStatusByName(lockName);
            if (clusterLockStatus == null) {
                clusterLockDao.insertEmptyClusterLock(lockName);
                return clusterLockDao.getClusterLockStatusByName(lockName);
            } else {
                return clusterLockStatus;
            }
        }, 100, TimeUnit.MILLISECONDS);
    }

    @Nonnull
    @Override
    public String getName()
    {
        return lockName;
    }

    @Override
    public boolean isLocked() {
        return getClusterLockStatus().getLockedByNode() != null;
    }

    boolean isLockedLocally() {
        Owner owner = ownerRef.get();
        return owner != null && owner.getThread() != null && owner.getThread().isAlive();
    }

    public void interruptOwner() {
        // general ugliness stems from all the AtomicReferences and WeakReferences

        Owner owner = ownerRef.get();
        if (owner != null) {
            Thread ownerThread = owner.getThread();
            if (ownerThread != null) {
                ownerThread.interrupt();
            }
        }
    }

    @Override
    public void lock()
    {
        final long startedAt = nowInMillis();
        final boolean wasInterrupted = Thread.interrupted();

        if (!tryLock())
        {
            stats.tallyWaitBegin();
            try
            {
                uninterruptibleWait();
            }
            finally
            {
                stats.tallyWaitEndAfter(nowInMillis() - startedAt);
            }
        }

        interruptIf(wasInterrupted);
    }

    private void uninterruptibleWait()
    {
        boolean wasInterrupted = false;

        int sleepTimeMillis = INITIAL_SLEEP_MILLIS;
        do
        {
            try
            {
                sleep(sleepTimeMillis);
            }
            catch (InterruptedException ie)
            {
                wasInterrupted = true;
            }

            // back off by increasing the sleep time
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, MAX_SLEEP_MILLIS);
        }
        while (!tryLock());

        interruptIf(wasInterrupted);
    }



    @Override
    public void lockInterruptibly() throws InterruptedException
    {
        final long startedAt = nowInMillis();
        if (Thread.interrupted())
        {
            throw new InterruptedException();
        }
        if (tryLock())
        {
            return;
        }

        stats.tallyWaitBegin();
        try
        {
            interruptibleWait();
        }
        finally
        {
            stats.tallyWaitEndAfter(nowInMillis() - startedAt);
        }
    }

    private void interruptibleWait() throws InterruptedException
    {
        int sleepTimeMillis = INITIAL_SLEEP_MILLIS;
        do
        {
            sleep(sleepTimeMillis);

            // back off by increasing the sleep time
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, MAX_SLEEP_MILLIS);
        }
        while (!tryLock());
    }

    /**
     * Attempts to acquire ownership of the lock, first locally as a thread and, if that succeeds, in DB storage, as a cluster node(by {@link ClusterLockDao#tryAcquireLock(String)}).
     * Lock can be acquired in one of the following situations:
     * <ol>
     *     <li>No other thread owns this lock at the beginning of the attempt AND attempt to acquire lock in DB results with success </li>
     *     <li>Other thread owns the lock but it's dead AND attempt to acquire lock in DB results with success  </li>
     *     <li>Lock is already owned by the same thread(in that case only internal call depth counter is increased)</li>
     * </ol>
     *
     * Once lock is owned, a recurring job is registered in {@link DatabaseClusterLockLeaseRenewer} that refreshes locks time of last update by  {@link ClusterLockDao#renewLease(String)}.
     * Renewal job repeats until either top level (i.e. non-reentrant) {@link DatabaseClusterLock#unlock()} or until it detects that local or cluster level ownership was lost(for example because the nodes VM frozen, locks ownership expired and other lock intercepted it's ownership).
     *
     *
     * @return true if ownership of the lock was sucessfully acquired, false if not.
     */
    @Override
    public boolean tryLock() {
        try {
            log.trace("Attempt to get cluster lock '{}' by {}.", lockName, Thread.currentThread());
            if (tryLockLocally()) {
                return Attempt.doTry(this::tryLockRemotely).withOnFalse(this::releaseThreadLock).withOnException(this::releaseThreadLock).go();
            }
            final Owner owner = ownerRef.get();
            if (tryReenterLock(owner)) {
                return true;
            }
            if (tryEvictDeadOwnerThread(owner)) {
                return Attempt.doTry(this::tryLockRemotely).withOnFalse(this::releaseThreadLock).withOnException(this::releaseThreadLock).go();
            }

            log.debug("Acquisition of cluster lock '{}' by {} failed. Lock is owned this node's '{}' ", lockName, Thread.currentThread(), owner.getThread());
            stats.tallyFailLocal();
            return false;
        } catch (IllegalMonitorStateException t) {
            stats.tallyStateError(nowInMillis());
            throw t;
        }
        catch (Throwable t) {
            stats.tallyError(nowInMillis());
            throw t;
        }
    }

    private boolean tryLockRemotely() {
        tryInsert();
        if (clusterLockDao.tryAcquireLock(lockName)) {
            databaseClusterLockLeaseRenewer.onLock(this);
            log.debug("Cluster lock '{}' was successfully acquired by this node's {}.", lockName, Thread.currentThread());
            depth.set(1);
            stats.tallyLockedAt(nowInMillis());
            return true;
        }
        log.debug("Acquisition of cluster lock '{}' by {} failed. Lock is owned by another node.", lockName, Thread.currentThread());
        stats.tallyFailRemote(nowInMillis());
        return false;
    }

    private void tryInsert() {
        if (!inserted.get()) {
            clusterLockDao.insertEmptyClusterLock(lockName);
            inserted.compareAndSet(false, true);
        }
    }

    private boolean tryLockLocally() {
        return ownerRef.compareAndSet(null, new Owner(Thread.currentThread()));
    }

    private boolean tryReenterLock(final Owner owner)
    {
        final Thread me = Thread.currentThread();
        if (owner.getThread() != me) {
            return false;
        }

        final int currentDepth = depth.incrementAndGet();
        log.trace("Cluster lock '{}' was successfully reentered by '{}', depth increased to {}", lockName, me, currentDepth);
        if (currentDepth < 0)  // Sign overflow?!
        {
            depth.decrementAndGet();
            throw new IllegalMonitorStateException("Maximum lock count exceeded");
        }
        return true;
    }

    private boolean isLockedByDeadThread(final Owner owner)
    {
        final Thread ownerThread = owner.getThread();
        if (ownerThread == null || !ownerThread.isAlive())
        {
            return true;
        }

        log.debug("Cluster lock '{}' currently held by another local thread '{}'.", lockName,
                ownerThread.getName());
        return false;
    }

    private boolean tryEvictDeadOwnerThread(final Owner owner) {
        if (isLockedByDeadThread(owner)) {
            final Thread me = Thread.currentThread();
            if (ownerRef.compareAndSet(owner, new Owner(me))) {
                log.error("During attempt to acquire lock '{}' by '{}' ownership by dead thread was detected. '{}' terminated before unlocking. Evicting previous owner and attempting to confirm ownership in DB...", lockName, me, owner);
                stats.tallyForcedUnlock();
                return true;
            }
        }
        return false;
    }

    /**
     * Removes current thread and cluster node from being owner of the lock, if it was already unlocked for all reentries. Otherwise, only decreases ownership depth by one level.
     * <p>
     * Assumes that the local ownership of the lock is still valid.
     * </p>
     * <p>
     * <p>
     * Releasing ownership of the lock consists of the following actions:
     * <ul>
     * <li>Deregisters renewal job for this lock from {@link DatabaseClusterLockLeaseRenewer}</li>
     * <li>attempts to release locks as the owner node within cluster by {@link ClusterLockDao#unlock(String)}</li>
     * <li>releases lock as the owning thread locally </li>
     * </ul>
     * </p>
     * <p>
     * If attempts to release lock in the DB resulting in an exception other than {@link IllegalMonitorStateException},  are retried a number of times, and the exception is rethrown if they all fail.
     *
     * @throws IllegalMonitorStateException if lock is detected to be no longer owned locally
     */
    @Override
    public void unlock() {
        final Thread me = Thread.currentThread();
        assertLocallyOwned(me);
            if (!tryExitReenteredLock()) {
                // We hit depth 0, so release it for real
                synchronized (monitor) {
                    log.trace("Cluster lock '{}' attempting unlock by '{}'", lockName, me);
                    boolean success = false;
                    int retries = 0;
                    try {
                        do {
                            try {
                                databaseClusterLockLeaseRenewer.onUnlock(this);
                                clusterLockDao.unlock(lockName);
                                success = true;
                                log.debug("Cluster lock '{}' unlocked by '{}'", lockName, me);
                            } catch (IllegalMonitorStateException ex) {
                                // We cannot unlock as this lock is held by somebody else.
                                final String errorReport = StatisticsHolder.getStatisticsSummary(getStatistics(), nowInMillis());
                                log.error("Cluster lock '{}' by '{}' unlock failed, held by someone else. " + errorReport, lockName, me);
                                stats.tallyStateError(nowInMillis());
                                break;
                            } catch (Exception ex) {
                                retries++;
                                if (retries > MAX_RETRIES) {
                                    final String errorReport = StatisticsHolder.getStatisticsSummary(getStatistics(), nowInMillis());
                                    log.error("Unable to unlock " + this.toString() + ", Number of retries exceeded, rethrowing ." + errorReport, ex);
                                    stats.tallyError(nowInMillis());
                                    throw ex;
                                }
                                log.error("Unable to unlock " + this.toString() + ", retrying.", ex);
                                stats.tallyError(nowInMillis());
                                try {
                                    sleep(INITIAL_SLEEP_MILLIS);
                                } catch (InterruptedException ie) {
                                    Thread.currentThread().interrupt();
                                }
                            }
                        } while (!success && !Thread.currentThread().isInterrupted());
                    } finally {
                        unlockLocally();
                    }
                }
            }
    }

    boolean renew() {
        synchronized (monitor) {
            if (Thread.currentThread().isInterrupted()) {
                return false;
            }
            if (isLockedLocally()) {
                final boolean b = tryRenew();
                return b;
            } else {
                throw new DeadOwnerThreadException();
            }
        }
    }

    private boolean tryRenew() {
        try {
            renewInDB();
            return true;
        } catch (IllegalMonitorStateException ex) {
            reportRenewStateError(ex);
            interruptOwner();
            return tryReacquireRemoteLockSafely();
        } catch (Throwable t) {
            reportRenewGeneralError(t);
        }
        return false;
    }

    private void reportRenewGeneralError(final Throwable t) {
        final String errorReport = StatisticsHolder.getStatisticsSummary(getStatistics(), nowInMillis());
        log.error("Failed to renew lease on lock: , " + getName() + ". Error occured during attempt to renew this lock in the database. Will retry on next scheduled run..." + errorReport, t);
        stats.tallyError(nowInMillis());
    }

    private void reportRenewStateError(final IllegalMonitorStateException ex) {
        final String errorReport = StatisticsHolder.getStatisticsSummary(getStatistics(), nowInMillis());
        log.error("Failed to renew lease on lock: " + getName() + ". this lock is no longer owned by this node in the database, attempting to reacquire it... " + errorReport, ex);
        stats.tallyStateError(nowInMillis());

    }

    private boolean tryReacquireRemoteLockSafely() {
        try {
            if (clusterLockDao.tryAcquireLock(getName())) {
                log.warn("Successfully reacquired lock: " + getName() + " after it was lost it for some time.");
                return true;
            } else {
                log.error("Failed to reacquire lock: " + getName() + ". Will retry until success or local unlock on next scheduled renewer runs...");
                return false;
            }
        } catch (final Throwable e) {
            log.error("Error during attempt to reacquire lock: " + getName() + ". Will retry until success or local unlock on next scheduled renewer runs...", e);
        }
        return false;
    }

    private void renewInDB() {
        clusterLockDao.renewLease(getName());
        stats.tallyRenewed(nowInMillis());
    }

    private boolean tryExitReenteredLock() {
        // exiting one level of a re-entered lock
        final int currentDepth = depth.decrementAndGet();
        if (currentDepth > 0) {
            log.trace("Reentered Cluster lock '{}' depth decremented by '{}' to {}", lockName, Thread.currentThread(), currentDepth);
            return true;
        }
        return false;
    }

    private void assertLocallyOwned(final Thread me) {
        final Owner owner = ownerRef.get();
        if (owner == null || owner.getThread() != me) {
            throw new IllegalMonitorStateException("Cluster lock '" + lockName + "' cannot be unlocked because it is not owned by this thread: " + me + " (owner: " + ((owner == null || owner.getThread() == null) ? "null" : owner.getThread().getName()) + ")");
        }
    }



    @Override
    public boolean tryLock(long waitTime, @Nonnull TimeUnit unit) throws InterruptedException {
        // By contract we test for interruption before we try acquire the lock

        if (Thread.interrupted()) {
            throw new InterruptedException();
        }

        final long startedAt = nowInMillis();
        if (tryLock()) {
            return true;
        }

        final long deadline = startedAt + unit.toMillis(waitTime);
        stats.tallyWaitBegin();
        try {
            return tryLockWaitWithTimeout(deadline);
        } finally {
            stats.tallyWaitEndAfter(nowInMillis() - startedAt);
        }
    }

    private boolean tryLockWaitWithTimeout(final long deadline) throws InterruptedException {
        long sleepTimeMillis = INITIAL_SLEEP_MILLIS;
        do {
            // Someone else has the lock right now
            final long remainingWaitTime = deadline - nowInMillis();

            // check for timeout
            if (remainingWaitTime <= 0) {
                return false;
            }

            // Wait a while before we try again ... but don't sleep past the timeout
            sleepTimeMillis = Math.min(sleepTimeMillis, remainingWaitTime);
            sleep(sleepTimeMillis);

            // back off by increasing the next sleep time
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, MAX_SLEEP_MILLIS);
        } while (!tryLock());

        return true;
    }

    private void releaseThreadLock() {
        final Owner currentOwner = ownerRef.get();
        if (Thread.currentThread().equals(currentOwner.getThread())) {
            ownerRef.compareAndSet(currentOwner, null);
        }
    }

    ;

    private void unlockLocally() {
        releaseThreadLock();
        stats.tallyUnlockedAt(nowInMillis());
    }

    @VisibleForTesting
    long nowInMillis() {
        return System.currentTimeMillis();
    }

    @Override
    public boolean isHeldByCurrentThread() {
        final Owner owner = ownerRef.get();
        return owner != null && owner.getThread() == Thread.currentThread();
    }


    @Nonnull
    @Override
    public Condition newCondition()
    {
        throw new UnsupportedOperationException("newCondition() not supported in ClusterLock");
    }

    /**
     * Returns the current lock status from the DB.
     *
     * @return the current lock status from the DB.
     */
    @Nonnull
    public ClusterLockStatus getClusterLockStatus()
    {
        // most the time the row should already exist in the DB
        final ClusterLockStatus lock = databaseLockStatusSupplier.get();
        return lock;
    }



    /**
     * {@inheritDoc}
     * <p>
     * The statistics for database cluster locks have the following quirks to them:
     * </p>
     * <ul>
     * <li>{@link StatisticsKey#AVERAGE_HOLD_TIME_MILLIS AVERAGE_HOLD_TIME_MILLIS} and
     *          {@link StatisticsKey#AVERAGE_WAIT_TIME_MILLIS AVERAGE_WAIT_TIME_MILLIS} &mdash; These
     *          do not update until the hold or wait completes; they do not reflect any time from a currently
     *          existing hold or wait on the lock.  Any lock that is forcibly cleared for any reason is excluded
     *          from the average hold time.</li>
     * <li>{@link StatisticsKey#FAIL_LOCAL FAIL_LOCAL} and {@link StatisticsKey#FAIL_REMOTE FAIL_REMOTE} and
     *          {@link StatisticsKey#LAST_FAIL_REMOTE LAST_FAIL_REMOTE} &mdash; These
     *          include the internal activity of long calls to {@link #lock()}, {@link #lockInterruptibly()}, and
     *          {@link #tryLock(long, TimeUnit)}.  Each of these methods is implemented by calling {@link #tryLock()}
     *          multiple times until it is successful, the request times out, or an {@code InterruptedException}
     *          is thrown (as the method contracts permit).  Although the wait queue length and average wait time
     *          are managed in terms of the overarching request, each failed poll is recorded as an individual
     *          failure to acquire the lock.</li>
     * <li>{@link StatisticsKey#FORCED_UNLOCK FORCED_UNLOCK}  &mdash; Includes all successful attempts to unlock thread owned by another dead thread</li>
     * <li>{@link StatisticsKey#LAST_ACCESS LAST_ACCESS} &mdash; Follows the most recent database-level
     *          state change attempt(lock/unlock) or confirmation attempt(renewal). In some cases (such as a reentrance, a request like {@link #tryLock()} may not need
     *          to access the database, so this value is not guaranteed to be updated by every request.</li>
     * <li>{@link StatisticsKey#ERROR ERROR} and {@link StatisticsKey#LAST_ERROR LAST_ERROR} &mdash; Includes all dao requests that have failed because they were
     *            terminated by an Exception other than {@link IllegalStateException}</li>
     * <li>{@link StatisticsKey#STATE_ERROR STATE_ERROR} and {@link StatisticsKey#LAST_STATE_ERROR LAST_STATE_ERROR} &mdash; Includes all dao requests that have failed because they were
     *            terminated by an {@link IllegalStateException}</li>
     * </ul>
     */
    @Nonnull
    @Override
    public Map<StatisticsKey, Long> getStatistics()
    {
        return stats.getStatistics();
    }

    private static class Attempt {

        private Optional<Runnable> onFalse = Optional.empty();
        private Optional<Runnable> onException = Optional.empty();
        private BooleanSupplier action;

        private Attempt(final Attempt builder) {
            onFalse = builder.onFalse;
            onException = builder.onException;
            action = builder.action;
        }

        private Attempt(BooleanSupplier action) {
            this.action = action;
        }

        public static Attempt doTry(BooleanSupplier action) {
            return new Attempt(action);
        }

        private boolean perform() {
            try {
                final boolean result = action.getAsBoolean();
                if (!result) {
                    onFalse.ifPresent(Runnable::run);
                    return false;
                }
                return result;
            } catch (Exception ex) {
                onException.ifPresent(Runnable::run);
                throw ex;
            }
        }

        public Attempt withOnFalse(final Runnable onFailure) {
            this.onFalse = Optional.of(onFailure);
            return this;
        }

        public Attempt withOnException(final Runnable onFailure) {
            this.onException = Optional.of(onFailure);
            return this;
        }

        public boolean go() {
            return new Attempt(this).perform();
        }
    }

    @VisibleForTesting
    void sleep(long timeout) throws InterruptedException
    {
        Thread.sleep(timeout);
    }

    private static void interruptIf(final boolean wasInterrupted)
    {
        if (wasInterrupted)
        {
            Thread.currentThread().interrupt();
        }
    }

    static class Owner
    {
        private final WeakReference<Thread> thd;
        private final String name;

        Owner(final Thread me)
        {
            this.thd = new WeakReference<Thread>(me);
            this.name = me.getName();
        }

        Thread getThread()
        {
            return thd.get();
        }

        @Override
        public String toString()
        {
            return name;
        }
    }
}
