package com.atlassian.beehive;

import javax.annotation.Nonnull;

import com.atlassian.annotations.PublicApi;

/**
 * Provides methods for running code under a cluster-wide lock.
 * <p>
 *   For safety reasons, an acquired lock will never time-out. However if the node that holds the lock is shutdown or is
 *   unable to reach the DB for an extended period then the lock is considered to no longer be valid and can be acquired
 *   by another node.
 * </p>
 * <p>
 *     The Locks returned by this service are re-entrant.
 * </p>
 * <p>
 *     A simple implementation of this Service can be provided by single-node apps in order to satisfy cross-product libraries that need it.
 * </p>
 * <p>
 *     Lock names can be up to 255 characters.
 *     Obviously they should be namespaced adequately to avoid conflicts with other plugins - the reverse DNS name convention is recommended.
 *     Some implementations will store the lock name in the underlying database, and so the name may be treated as case-sensitive
 *     in some environments, and case-insensitive in other environments.
 *     To avoid unforeseen compatibility issues, consumers are recommended to use names that consist of "simple" ASCII
 *     characters (eg alphanumeric, hyphen and underscore).
 * </p>
 * <p>
 * Application developers should consider using one of the implementations provided by {@code beehive-core} or
 * {@code beehive-hazelcast} instead of creating an implementation that is specific to the application.
 * </p>
 * <ul>
 * <li><code><strong>SimpleClusterLockService</strong> (beehive-core)</code> &mdash; for single node applications</li>
 * <li><code><strong>DatabaseClusterLockService</strong> (beehive-core)</code> &mdash; for clustered applications that
 *          use database objects to control cluster-wide locking behaviour</li>
 * <li><code><strong>HazelcastClusterLockService</strong> (beehive-hazelcast)</code> &mdash; for clustered applications
 *          that can use the Hazelcast library to implement distributed messaging and locking</li>
 * </ul>
 */
@PublicApi
public interface ClusterLockService
{
    /**
     * Returns the cluster-wide lock for the given name.
     *
     * @param lockName the globally unique lock identifier
     * @return the cluster-wide lock for the given name.
     */
    ClusterLock getLockForName(@Nonnull String lockName);
}
