package com.atlassian.bamboo.plugins.tomcat.tasks;

import com.atlassian.bamboo.build.logger.BuildLogger;
import com.atlassian.bamboo.plugins.tomcat.configuration.DeployAppConfigurator;
import com.atlassian.bamboo.plugins.tomcat.manager.TaskTomcatConnection;
import com.atlassian.bamboo.plugins.tomcat.manager.TomcatApplicationManager;
import com.atlassian.bamboo.plugins.tomcat.manager.TomcatApplicationManagerImpl;
import com.atlassian.bamboo.plugins.tomcat.manager.TomcatConnection;
import com.atlassian.bamboo.plugins.tomcat.manager.TomcatResult;
import com.atlassian.bamboo.security.EncryptionService;
import com.atlassian.bamboo.task.CommonTaskContext;
import com.atlassian.bamboo.task.CommonTaskType;
import com.atlassian.bamboo.task.TaskException;
import com.atlassian.bamboo.task.TaskResult;
import com.atlassian.bamboo.task.TaskResultBuilder;
import com.atlassian.bamboo.variable.CustomVariableContext;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;

/**
 * Deploys a war on the local file system to Tomcat. Will stop
 */
public class DeployAppTask implements CommonTaskType
{
    private static final Logger log = Logger.getLogger(DeployAppTask.class);
    // ------------------------------------------------------------------------------------------------------- Constants
    // ------------------------------------------------------------------------------------------------- Type Properties
    // ---------------------------------------------------------------------------------------------------- Dependencies
    private final EncryptionService encryptionService;
    private final CustomVariableContext customVariableContext;

    // ---------------------------------------------------------------------------------------------------- Constructors
    public DeployAppTask(final EncryptionService encryptionService, final CustomVariableContext customVariableContext)
    {
        this.encryptionService = encryptionService;
        this.customVariableContext = customVariableContext;
    }
    // ----------------------------------------------------------------------------------------------- Interface Methods

    @NotNull
    @java.lang.Override
    public TaskResult execute(@NotNull final CommonTaskContext taskContext) throws TaskException
    {
        final BuildLogger buildLogger = taskContext.getBuildLogger();
        final TomcatConnection connection = new TaskTomcatConnection(taskContext, encryptionService);
        final TomcatApplicationManager tomcatManager = new TomcatApplicationManagerImpl(connection, taskContext, customVariableContext, buildLogger);
        final TaskResultBuilder taskResultBuilder = TaskResultBuilder.newBuilder(taskContext);

        final String context = taskContext.getConfigurationMap().get(DeployAppConfigurator.APP_CONTEXT);
        final String version = taskContext.getConfigurationMap().get(DeployAppConfigurator.APP_VERSION);
        final String warFilePath = taskContext.getConfigurationMap().get(DeployAppConfigurator.WAR_FILE_PATH);
        final String deploymentTag = taskContext.getConfigurationMap().get(DeployAppConfigurator.DEPLOYMENT_TAG);

        final File file = new File(taskContext.getRootDirectory(), warFilePath);
        if (file.isFile())
        {
            try
            {
                final StringBuilder deployMessage = new StringBuilder()
                        .append("Deploying application with war file '")
                        .append(warFilePath)
                        .append("' to context '")
                        .append(context)
                        .append("' to server '")
                        .append(connection.getURL())
                        .append("'");

                if (StringUtils.isNotEmpty(version))
                {
                    deployMessage.append(" with version '" + version + "'");
                }

                if (StringUtils.isNotEmpty(deploymentTag))
                {
                    deployMessage.append(" with tag '" + deploymentTag + "'");
                }

                buildLogger.addBuildLogEntry(deployMessage.toString());

                final TomcatResult result = tomcatManager.deployApplication(context, version, deploymentTag, file);
                if (result.isSuccessful())
                {
                    buildLogger.addBuildLogEntry("Application was successfully deployed.");
                    taskResultBuilder.success();
                }
                else
                {
                    final String message = "Application failed to deploy: " + result.getReason();
                    buildLogger.addErrorLogEntry(message);
                    taskResultBuilder.failed();
                }
            }
            catch (IOException e)
            {
                final String log = "Could not deploy application: " + e.getMessage();
                buildLogger.addErrorLogEntry(log, e);
                taskResultBuilder.failedWithError();
            }
        }
        else
        {
            final String noWarMessage = "Could not find war file at " + file.getAbsolutePath() + ". The application could not be deployed.";
            buildLogger.addErrorLogEntry(noWarMessage);
            taskResultBuilder.failedWithError();
        }

        return taskResultBuilder.build();
    }

    // -------------------------------------------------------------------------------------------------- Action Methods
    // -------------------------------------------------------------------------------------------------- Public Methods
    // -------------------------------------------------------------------------------------- Basic Accessors / Mutators
}
