define('feature/github-repository-form', [
    'jquery',
    'underscore',
    'util/ajax-form',
    'util/events'
], function(
    $,
    _,
    AjaxForm,
    events
) {
    'use strict';

    /**
     * GitHub Repository form
     * @param {Object} opts - options with which to set up the form
     * @constructor
     */
    function RepositoryForm(opts) {
        var defaults = {
            repositoryKey: null,
            repositoryId: 0,
            selectors: {
                container: null,
                repositoryType: null,
                username: null,
                password: null,
                loadRepositoriesButton: null,
                repository: null,
                branch: null
            },
            inheritedData: {
                repository: '',
                id: ''
            }
        };

        var options = $.extend(true, defaults, opts);

        // Values
        this.repositoryKey = options.repositoryKey;
        this.repositoryId = options.repositoryId;

        // jqXHR objects for repository/branch retrieval
        this._jqXHRRepository = null;
        this._jqXHRBranch = null;

        // Form controls
        this.$container = $(options.selectors.container);
        this.$repositoryType = $(options.selectors.repositoryType);
        this.$username = this.$container.find(options.selectors.username);
        this.$password = this.$container.find(options.selectors.password);
        this.$loadRepositoriesButton = this.$container.find(options.selectors.loadRepositoriesButton);
        this.$repository = this.$container.find(options.selectors.repository);
        this.$branch = $(options.selectors.branch);
        this.$form = this.$username.closest('form');

        this.inheritedData = options.inheritedData;
        this.$overrideRepositoryLocation = $('#overrideRepositoryLocation');

        events.EventBus.on('repository:server:toggle-override', this.resetRepository.bind(this));
    }

    RepositoryForm.prototype = {
        init: function() {
            this.$repository.select2({enable: false, width: 'copy'});
            // Attach handlers
            this.$loadRepositoriesButton.click(_.bind(this.loadRepositoriesClick, this));
            this.$repository.change(_.bind(this.repositoryChange, this));

            // Load repository list on load
            if (this.$repositoryType.val() === this.repositoryKey &&
                (this.repositoryId || (this.$username.val() && this.$password.val()))) {
                this.loadRepositories();

                if (this.$repository.val() !== null) {
                    this.loadBranches(true);
                }
            }
        },
        loadRepositoriesClick: function() {
            this.clearFieldErrors();
            this.$repository.prop('disabled', true);
            this.$branch.prop('disabled', true);
            if (this._jqXHRBranch) { this._jqXHRBranch.abort(); }
            this.loadRepositories().done(_.bind(function(json) {
                var repositories = json.repositories;

                if (json.status === 'OK' && repositories && repositories.length) {
                    this.$repository.focus();
                    if (!this.$branch.children().length) {
                        this.$branch.append(this.generateBranchOption(this.defaultBranch()));
                    }

                    // select first repository by default
                    this.$repository.val(this.$repository.find('option:first').val());

                    this.loadBranches(this.isOverrideOn());
                }
            }, this));
        },
        isOverrideOn: function() {
            return !this.$overrideRepositoryLocation.length || this.$overrideRepositoryLocation.val() === 'true';
        },
        resetRepository: function() {
            var overrideOn = this.isOverrideOn();
            if (this.inheritedData.repository && this.inheritedData.repository !== this.$repository.val()) {
                this.clearFieldErrors();
                this.$branch.prop('disabled', true).empty().append(this.generateBranchOption(this.defaultBranch()));
                this.loadBranches(overrideOn);
            }
        },
        repositoryChange: function() {
            this.clearFieldErrors();
            this.$branch.prop('disabled', true).empty().append(this.generateBranchOption(this.defaultBranch()));
            this.loadBranches(true);
        },
        loadRepositories: function() {
            var $repository = this.$repository;
            var $loadingOptgroup = $('<optgroup />')
                .attr('label', AJS.I18n.getText('repository.github.loadingRepositories')).appendTo($repository);
            var currentlySelected = $repository.val();
            var generateRepositoryOption = this.generateRepositoryOption;
            var update = _.bind(function(json, textStatus, jqXHR) {
                var $list = document.createDocumentFragment();
                var repositories = json.repositories;

                if (json.status === 'OK' && repositories && repositories.length) {
                    $.each(repositories, function() {
                        $list.appendChild(generateRepositoryOption(this).get(0));
                    });

                    $repository.empty().prop('disabled', false).get(0).appendChild($list);
                    $repository.val(currentlySelected);
                } else {
                    showError(jqXHR);
                }
            }, this);

            var restore = _.bind(function() {
                $loadingOptgroup.remove();
                this.$loadRepositoriesButton.prop('disabled', false);
            }, this);

            var showError = _.bind(function(jqXHR) {
                var response;
                var errors = [];
                var repositories;
                var fieldErrors = {};

                if (jqXHR.statusText !== 'abort') {
                    try {
                        response = $.parseJSON(jqXHR.responseText);
                        errors = response.errors || [];
                        repositories = response.repositories;
                        fieldErrors = response.fieldErrors || {};
                    } catch (e) {}

                    if (repositories && !repositories.length) {
                        errors.push(AJS.I18n.getText('repository.github.error.noRepositories').replace('{0}',
                            this.$username.val()));
                    }
                    if (!errors.length) {
                        errors.push(AJS.I18n.getText('repository.github.ajaxError') +
                            '[' + jqXHR.status + ' ' + jqXHR.statusText + ']');
                    }
                    this.addFieldErrors(this.$repository, errors, fieldErrors);
                }
            }, this);

            this.$loadRepositoriesButton.prop('disabled', true);
            return this.getRepositoryList().done(update).fail(showError).always(restore);
        },
        generateRepositoryOption: function(repository) {
            return $('<option/>', {
                text: repository.full_name,
                val: repository.full_name,
                data: repository
            });
        },
        loadBranches: function(overrideOn) {
            var $branch = this.$branch;
            var $loadingOptgroup = $('<optgroup />')
                .attr('label', AJS.I18n.getText('repository.github.loadingBranches')).appendTo($branch);
            var $container = $branch.closest('.field-group');
            var currentlySelected = $branch.val();
            var generateBranchOption = this.generateBranchOption;
            var update = _.bind(function(json, textStatus, jqXHR) {
                var $list = $();
                var branches = json.branches;

                if (json.status === 'OK' && branches) {
                    $.each(branches, function() {
                        $list = $list.add(generateBranchOption(this.name));
                    });
                    $branch.empty().append($list).prop('disabled', false).val(currentlySelected);
                    if (currentlySelected && $branch.val() !== currentlySelected) {
                        $branch.val(this.defaultBranch());
                    }
                } else {
                    showError(jqXHR);
                }
            }, this);

            var restore = _.bind(function() { $loadingOptgroup.remove(); }, this);
            var showError = _.bind(function(jqXHR) {
                var response;
                var errors = [];
                var fieldErrors = {};

                if (jqXHR.statusText !== 'abort') {
                    try {
                        response = $.parseJSON(jqXHR.responseText);
                        errors = response.errors || [];
                        fieldErrors = response.fieldErrors || {};
                    } catch (e) {}

                    if (!errors.length) {
                        errors.push(AJS.I18n.getText('repository.github.ajaxError') +
                            '[' + jqXHR.status + ' ' + jqXHR.statusText + ']');
                    }
                    this.addFieldErrors(this.$branch, errors, fieldErrors);
                }
            }, this);

            // Show branches field if it was hidden
            if ($container.hasClass('hidden')) {
                $container.hide().removeClass('hidden').slideDown();
            }
            return this.getBranchList(this.$repository.val(), overrideOn).done(update).fail(showError).always(restore);
        },
        generateBranchOption: function(branch) {
            return $('<option/>', { text: branch });
        },
        defaultBranch: function() {
            return 'master';
        },
        getJsonResponse: function(url, data, jqXHR, $field) {
            var $container = $field.closest('.field-group');
            var loadingClass = 'loading';

            if (jqXHR) { jqXHR.abort(); }

            $container.addClass(loadingClass);

            return $.ajax({
                type: 'POST',
                url: url,
                data: JSON.stringify(data),
                contentType: 'application/json',
                processData: false,
                dataType: 'json'
            }).always(function() { $container.removeClass(loadingClass); });
        },
        getRepositoryList: function() {
            var url = AJS.contextPath() + '/rest/git/latest/gh/repositories/' + this.$username.val() + '/';

            this._jqXHRRepository = this.getJsonResponse(url, {
                password: this.$password.val(),
                repositoryId: this.repositoryId
            }, this._jqXHRRepository, this.$repository);

            return this._jqXHRRepository;
        },
        getBranchList: function(repository, overrideOn) {
            var url = AJS.contextPath() + '/rest/git/latest/gh/repositories/';
            if (overrideOn) {
                url = url + repository + '/branches/';
                this._jqXHRBranch = this.getJsonResponse(url, {
                    username: this.$username.val(),
                    password: this.$password.val(),
                    repository: repository,
                    repositoryId: this.repositoryId
                }, this._jqXHRBranch, this.$branch);
            } else {
                // use inherited data
                url = url + this.inheritedData.repository + '/branches/';
                this._jqXHRBranch = this.getJsonResponse(url, {
                    repository: this.inheritedData.repository,
                    repositoryId: this.inheritedData.repositoryId
                }, this._jqXHRBranch, this.$branch);
            }

            return this._jqXHRBranch;
        },
        addFieldErrors: function($field, errors, fieldErrors) {
            if (errors && errors.length) {
                var fieldName = $field.attr('name');
                fieldErrors[fieldName] = fieldErrors[fieldName] || [];
                errors.forEach(function(error) {
                    fieldErrors[fieldName].push(error);
                });
            }
            new AjaxForm(this.$form).injectErrors({ errors: [], fieldErrors: fieldErrors });
        },
        clearFieldErrors: function() {
            this.$form.find('.error').remove();
        }
    };

    return RepositoryForm;
});
