define('plugin-bitbucket/repository-autocomplete', [
    'jquery',
    'underscore',
    'widget/autocomplete-ajax',
    'plugin-bitbucket/bitbucket-autocomplete',
    'widget/help-dialog',
    'util/events',
    'util/elements',
    'util/ajax'
], function(
    $,
    _,
    AutocompleteAjax,
    BitbucketAutocomplete,
    HelpDialog,
    events,
    elements,
    ajax
) {
    'use strict';

    var RepositoryAutocomplete = BitbucketAutocomplete.extend({

        mixins: [
            events.EventBusMixin,
            elements
        ],

        initialize: function(options) {
            this.params = $.extend({
                url: AJS.contextPath() + '/rest/hg/latest/bb/repositories',
                urlSingle: AJS.contextPath() + '/rest/hg/latest/bb/repository'
            }, options.params || {});

            this.query = $.extend({
                limit: 10
            }, options.query || {});

            var dataCallback = _.bind(function(term, page, cache) {
                var query = $.extend({
                    start: Math.max((page - 1), 0),
                    query: term
                }, this.query);

                if (!cache && (page - 1) > 0 && this.data.nextPage) {
                    query.start = this.data.nextPage;
                }
                _.extend(query, this.getJsonAuthenticationParams());
                return JSON.stringify(query);
            }, this);

            var settings = {
                minimumInputLength: 0,
                ajax: {
                    url: this.params.url,
                    type: 'POST',
                    dataType: 'json',
                    data: dataCallback,
                    params: {
                        error: _.bind(this.onRemoteError, this)
                    },
                    results: _.bind(function(data, page) {
                        return this.processData({
                            results: data.values,
                            nextPage: data.nextPageStart || 0,
                            more: !data.lastPage
                        });
                    }, this)
                },
                id: function(object) {
                    return object.fullName;
                }
            };

            this.onEvent('unbind', this.onUnbind);
            this.onEvent('bitbucket:repository:selection:init', this.onRepositorySelectionInit);
            this.onEvent('bitbucket:form:config', this.onFormConfig);

            this.$fieldGroup = this.$el.parents('.field-group:first');

            AutocompleteAjax.prototype.initialize.apply(
                this, [$.extend(settings, this.params)]
            );

            this.addDescriptionField();
            this.disable();
        },

        onUnbind: function() {
            this.undelegateEvents();
            this.$el.removeData().unbind();
            this.offEventNamespace('bitbucket');
        },

        onFormConfig: function(instance, config) {
            this.config = config;
        },

        onRepositorySelectionInit: function() {
            this.$fieldGroup.show();
            this.$fieldGroup.find('.field-value').show();
            this.clearRepositoryErrorMessages();
            this.$fieldGroup.find('.aui-message').remove();
            this.clearQueryCache().disable();

            this.toggleContainer(this.$fieldGroup.parent(), true)
                .promise().done(_.bind(function() {
                    this.addLoadingIcon();
                }, this));

            if (this.params.slugEl.val().length) {
                var repository = this.params.slugEl.val();
                var scm = this.params.scmEl.val().toLowerCase();
                this.getRemoteData(this.params.urlSingle, {
                    dataType: this.settings.ajax.dataType,
                    data: {repository: repository}
                }).done(
                    _.bind(function(data) {
                        this.triggerEvent('bitbucket:repository:init', data);
                        this.$el.auiSelect2('data', data);
                    }, this)
                ).fail(
                    _.bind(function() {
                        var localData = {
                            fullName: repository,
                            scm: scm
                        };
                        this.triggerEvent('bitbucket:repository:init', localData);
                        this.$el.auiSelect2('data', localData);
                    }, this)
                ).always(
                    _.bind(function() {
                        this.removeLoadingIcon().enable();
                    }, this)
                );
            } else {
                this.removeLoadingIcon().enable();
            }
        },

        onRegisterEvents: function() {
            this.$el.on('change', _.bind(this.onRepositoryChange, this));
        },

        onRepositoryChange: function(event) {
            if (event.val.length > 0) {
                this.triggerEvent('bitbucket:errors:clear');
                var repositoryData = event.added;
                this.triggerEvent('bitbucket:repository:update', repositoryData);
            }
        },

        onFormatResult: function(item) {
            if (!_.isEmpty(item)) {
                return bamboo.plugins.bitbucket.repository.autocomplete.row({
                    item: item
                });
            }
        },

        onFormatSelection: function(item) {
            if (!_.isEmpty(item)) {
                return bamboo.plugins.bitbucket.repository.autocomplete.row({
                    item: item
                });
            }
        },

        clearRepositoryErrorMessages: function() {
            this.$fieldGroup.find('.aui-message').remove();
        },

        addDescriptionField: function() {
            var $descriptionDiv = $(document.createElement('div'))
                .addClass('description');

            this.$el.parent('.field-value').after($descriptionDiv);
            return this;
        }
    });

    return RepositoryAutocomplete;
});
