define('plugin-bitbucket/branch-autocomplete', [
    'jquery',
    'underscore',
    'widget/autocomplete-ajax',
    'plugin-bitbucket/bitbucket-autocomplete',
    'util/events',
    'util/elements'
], function(
    $,
    _,
    AutocompleteAjax,
    BitbucketAutocomplete,
    events,
    elements
) {
    'use strict';

    var BranchAutocomplete = BitbucketAutocomplete.extend({

        mixins: [
            events.EventBusMixin,
            elements
        ],

        initialize: function(options) {
            this.params = $.extend({
                url: AJS.contextPath() + '/rest/hg/latest/bb/branches',
                urlSingle: AJS.contextPath() + '/rest/hg/latest/bb/branch',
                urlDefault: AJS.contextPath() + '/rest/hg/latest/bb/branches/default'
            }, options.params || {});

            this.query = {
                limit: 10
            };

            var dataCallback = _.bind(function(term, page, cache) {
                var query = $.extend({
                    start: Math.max((page - 1), 0),
                    query: term
                }, this.query);

                if (!cache && (page - 1) > 0 && this.data.nextPage) {
                    query.start = this.data.nextPage;
                }
                _.extend(query, this.getJsonAuthenticationParams());
                return JSON.stringify(query);
            }, this);

            var settings = {
                minimumInputLength: 0,
                ajax: {
                    type: 'POST',
                    url: this.params.url,
                    dataType: 'json',
                    data: dataCallback,
                    params: {
                        error: _.bind(this.onRemoteError, this)
                    },
                    results: _.bind(function(data, page) {
                        return this.processData({
                            results: data.values,
                            nextPage: data.nextPageStart || 0,
                            more: !data.lastPage
                        });
                    }, this)
                },
                id: function(object) {
                    return object.branch;
                }
            };

            this.onEvent('unbind', this.onUnbind);
            this.onEvent('bitbucket:repository:change', this.onRepositoryChange);
            this.onEvent('bitbucket:repository:selection:init', this.onRepositorySelectionInit);
            this.onEvent('bitbucket:form:config', this.onFormConfig);

            this.$fieldGroup = this.$el.parents('.field-group:first');

            AutocompleteAjax.prototype.initialize.apply(
                this, [$.extend(settings, this.params)]
            );

            this.disable();
        },

        onUnbind: function() {
            this.undelegateEvents();
            this.$el.removeData().unbind();
            this.offEventNamespace('bitbucket');
        },

        onFormConfig: function(instance, config) {
            this.config = config;
        },

        onRepositorySelectionInit: function() {
            this.clearQueryCache().clearValue().disable()
                .toggleContainer(this.$fieldGroup.parent(), false);
        },

        clearBranchErrorMessages: function() {
            this.$fieldGroup.find('.error').remove();
        },

        onRepositoryChange: function(instance, item, isInit) {
            this.$fieldGroup.show();
            this.$fieldGroup.find('.field-value').show();
            this.clearBranchErrorMessages();

            this.clearQueryCache().disable();

            this.query = {
                limit: 10
            };

            if (item && item.fullName) {
                this.query.repository = item.fullName;
            }

            if (isInit !== true) {
                this.$el.val('');
            }

            if (this.query.repository) {
                var value = this.$el.val();
                var callback = _.bind(function(data) {
                    this.triggerEvent('bitbucket:branch:change', data);
                    this.$el.auiSelect2('data', data);
                    this.removeLoadingIcon().enable();
                }, this);

                this.toggleContainer(this.$fieldGroup.parent(), true)
                    .promise().done(_.bind(function() {
                        this.addLoadingIcon();
                    }, this));

                if (!value.length) {
                    this.getRemoteData(this.params.urlDefault, {
                        dataType: this.settings.ajax.dataType,
                        data: this.query
                    })
                        .done(callback);
                } else {
                    this.query.branch = value;
                    this.getRemoteData(this.params.urlSingle, {
                        dataType: this.settings.ajax.dataType,
                        data: this.query
                    })
                        .always(function() {
                            callback({branch: value});
                        });
                }
            }
        },

        onRegisterEvents: function() {
            this.$el.on('change', _.bind(function(event) {
                if (event.val.length > 0) {
                    this.triggerEvent('bitbucket:errors:clear');
                    this.triggerEvent('bitbucket:branch:change', event.added);
                }
            }, this));
        },

        onFormatResult: function(item) {
            if (!_.isEmpty(item)) {
                return bamboo.plugins.bitbucket.branch.autocomplete.row({
                    item: item
                });
            }
        },

        onFormatSelection: function(item) {
            if (!_.isEmpty(item)) {
                return bamboo.plugins.bitbucket.branch.autocomplete.row({
                    item: item
                });
            }
        }
    });

    return BranchAutocomplete;
});
