/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.ActiveInstance;
import com.amazonaws.services.ec2.model.CreateTagsRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetInstancesResult;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.amazonaws.services.ec2.model.Tag;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.AWSException;
import com.atlassian.aws.ec2.EC2InstanceListener;
import com.atlassian.aws.ec2.EC2InstanceState;
import com.atlassian.aws.ec2.InstanceLaunchConfiguration;
import com.atlassian.aws.ec2.InstancePaymentType;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.InstanceStatus;
import com.atlassian.aws.ec2.InstanceTerminator;
import com.atlassian.aws.ec2.RemoteEC2Instance;
import com.atlassian.aws.ec2.awssdk.AbstractDelegatingInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSpotFleetActiveInstance;
import com.atlassian.aws.ec2.awssdk.AwsSpotFleetInstanceRequest;
import com.atlassian.aws.ec2.awssdk.AwsSpotInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.InstanceLauncher;
import com.atlassian.aws.ec2.awssdk.launch.InstanceLauncherFactory;
import com.atlassian.aws.ec2.caches.Ec2CacheMissException;
import com.google.common.collect.Iterables;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class RemoteEC2InstanceImpl
implements RemoteEC2Instance {
    private static final Logger log = Logger.getLogger(RemoteEC2InstanceImpl.class);
    static final int DEFAULT_TIMEOUT = 300;
    private static final int SHUTDOWN_TIMEOUT_IN_SECONDS = 300;
    private final AtomicInteger successiveSupervisionFailures = new AtomicInteger();
    private final int pollPeriodInSeconds;
    private final int maxSuccessiveSupervisionFailures;
    private final AWSAccount awsAccount;
    private final EC2InstanceListener listener;
    private final ScheduledExecutorService scheduledExecutorService;
    private final InstanceLaunchConfiguration instanceConfiguration;
    private final InstanceStatus instanceStatus;
    private static final long AWS_RESOURCE_ID_PROPAGATION_TIME = TimeUnit.SECONDS.toMillis(40L);
    private final InstanceLauncherFactory instanceLauncherFactory;
    private volatile InstanceReservationDescription lastKnownReservationDescription;
    private final AtomicBoolean startCalled = new AtomicBoolean();
    private final Runnable launcherTask = new CatchingRunnableDecorator("backgroundStart()", new Runnable(){

        @Override
        public void run() {
            RemoteEC2InstanceImpl.this.backgroundStart();
        }
    });
    private final Runnable supervisor = new CatchingRunnableDecorator("backgroundSupervise()", new Runnable(){

        @Override
        public void run() {
            RemoteEC2InstanceImpl.this.backgroundSupervise();
        }
    });
    private final AtomicBoolean isBeingTerminated = new AtomicBoolean();
    private volatile ScheduledFuture<?> supervisorJob;
    private volatile EC2InstanceState state = EC2InstanceState.INITIAL;

    public RemoteEC2InstanceImpl(@NotNull InstanceLaunchConfiguration instanceConfiguration, int pollPeriodInSeconds, int maxSuccessiveSupervisionFailures, EC2InstanceListener listener, AWSAccount awsAccount, ScheduledExecutorService scheduledExecutorService) {
        this.instanceConfiguration = instanceConfiguration;
        this.pollPeriodInSeconds = pollPeriodInSeconds;
        this.maxSuccessiveSupervisionFailures = maxSuccessiveSupervisionFailures;
        this.listener = listener;
        this.awsAccount = awsAccount;
        this.scheduledExecutorService = scheduledExecutorService;
        this.instanceStatus = new InstanceStatus();
        this.instanceLauncherFactory = new InstanceLauncherFactory(awsAccount);
    }

    @Override
    public void start() {
        if (!this.startCalled.compareAndSet(false, true)) {
            throw new IllegalStateException("Already started.");
        }
        this.scheduledExecutorService.execute(this.launcherTask);
    }

    @Override
    public void asyncTerminate() {
        if (!this.startCalled.get()) {
            throw new IllegalStateException("Not started.");
        }
        if (this.isBeingTerminated.compareAndSet(false, true)) {
            InstanceTerminator.terminate(this.awsAccount, this.scheduledExecutorService, this.instanceStatus);
        }
    }

    @Override
    public boolean isBeingTerminated() {
        return this.isBeingTerminated.get();
    }

    InstanceReservationDescription describeInstance() {
        InstanceReservationDescription newInstanceReservationDescription;
        boolean noInstanceId = this.instanceStatus.getInstanceId() == null;
        String spotOrFleetInstanceRequestId = this.instanceStatus.getSpotInstanceRequestId();
        if (noInstanceId && spotOrFleetInstanceRequestId == null) {
            throw new IllegalStateException("The instance has neither an instance id nor a spot request id");
        }
        this.lastKnownReservationDescription = newInstanceReservationDescription = noInstanceId ? this.describeSpotOrFleet(spotOrFleetInstanceRequestId) : this.describeInstance(this.lastKnownReservationDescription);
        return newInstanceReservationDescription;
    }

    private InstanceReservationDescription describeInstance(@Nullable InstanceReservationDescription lastKnownDescription) {
        try {
            Collection<Instance> describeInstancesResult = this.awsAccount.describeInstances(this.getInstanceId());
            Instance instance = (Instance)Iterables.getOnlyElement(describeInstancesResult);
            this.instanceStatus.setLaunchTime(instance.getLaunchTime());
            return new AwsInstanceReservationDescription(instance);
        }
        catch (Ec2CacheMissException e) {
            if (lastKnownDescription == null || lastKnownDescription.getState() != AwsSupportConstants.InstanceStateName.ShuttingDown) {
                throw e;
            }
            return new AbstractDelegatingInstanceReservationDescription(lastKnownDescription){

                @Override
                public AwsSupportConstants.InstanceStateName getState() {
                    return AwsSupportConstants.InstanceStateName.Terminated;
                }
            };
        }
    }

    @NotNull
    private InstanceReservationDescription describeSpotOrFleet(String spotOrFleetInstanceRequestId) {
        if (spotOrFleetInstanceRequestId.startsWith("sfr-")) {
            DescribeSpotFleetInstancesRequest describeSpotFleetInstancesRequest = new DescribeSpotFleetInstancesRequest().withSpotFleetRequestId(spotOrFleetInstanceRequestId);
            DescribeSpotFleetInstancesResult describeSpotFleetInstancesResult = this.awsAccount.getAmazonEc2().describeSpotFleetInstances(describeSpotFleetInstancesRequest);
            List activeInstances = describeSpotFleetInstancesResult.getActiveInstances();
            if (!activeInstances.isEmpty()) {
                ActiveInstance activeInstance = (ActiveInstance)Iterables.getOnlyElement((Iterable)activeInstances);
                return new AwsSpotFleetActiveInstance(activeInstance);
            }
            return new AwsSpotFleetInstanceRequest(spotOrFleetInstanceRequestId, null);
        }
        SpotInstanceRequest describedSpotRequest = this.describeSpotRequest();
        if (StringUtils.isNotBlank((CharSequence)describedSpotRequest.getInstanceId())) {
            describedSpotRequest.setState(AwsSupportConstants.InstanceStateName.SpotActive.toString());
            log.info((Object)("Spot instance request " + spotOrFleetInstanceRequestId + " is now active as instance " + describedSpotRequest.getInstanceId()));
        }
        return new AwsSpotInstanceReservationDescription(describedSpotRequest);
    }

    private SpotInstanceRequest describeSpotRequest() {
        Collection<SpotInstanceRequest> describeSpotInstanceRequestsResult = this.awsAccount.describeSpotInstanceRequests(this.instanceStatus.getSpotInstanceRequestId());
        return (SpotInstanceRequest)Iterables.getOnlyElement(describeSpotInstanceRequestsResult);
    }

    AWSException unexpectedStateException(InstanceReservationDescription instance) {
        return new AWSException("EC2 instance " + this.instanceStatus.getSensibleId() + " in an unexpected state " + instance.getStateDescription());
    }

    public void handleAddressChange(@NotNull InstanceReservationDescription instance) {
        String previousAddress = this.instanceStatus.getAddress();
        String newAddress = instance.getAddress();
        log.info((Object)("Bamboo has detected that the instance " + instance.getInstanceId() + ", previously available at " + previousAddress + " is now available at " + newAddress));
        this.instanceStatus.setAddressAndHostname(instance);
        this.listener.onInstanceAddressChange(previousAddress, newAddress);
    }

    public void handleStateChange(InstanceReservationDescription instance, AwsSupportConstants.InstanceStateName newState) {
        this.instanceStatus.setState(instance.getState());
        long timeElapsed = this.instanceStatus.getSecondsSinceStartupAttempt();
        String msg = null;
        switch (newState) {
            case SpotActive: {
                log.info((Object)("Spot request fulfilled after " + timeElapsed + " seconds."));
                this.updatePostLaunchData(instance);
                this.setSupervisionState(EC2InstanceState.PENDING, msg, null);
                break;
            }
            case SpotCancelled: 
            case SpotClosed: 
            case SpotFailed: {
                String statusDescription = "Spot request " + ((AwsSpotInstanceReservationDescription)instance).getSpotInstanceRequestId() + " state is " + (Object)((Object)newState) + " after " + timeElapsed + " seconds,";
                if (this.instanceConfiguration.isSpotRequestTimeoutExpired(timeElapsed)) {
                    log.info((Object)(statusDescription + ", falling back to a regular instance."));
                    this.launchInstance(this.instanceLauncherFactory.newOnDemandInstanceLauncher(this.instanceConfiguration, this.instanceStatus));
                    break;
                }
                log.info((Object)(statusDescription + ", assuming this was a manual cancellation."));
                this.setSupervisionState(EC2InstanceState.TERMINATED, msg, null);
                break;
            }
            case Running: {
                this.updatePostLaunchData(instance);
                String address = instance.getAddress();
                this.instanceStatus.setAddressAndHostname(instance);
                msg = "Bamboo has detected that EC2 instance " + this.getInstanceId() + " is now running at " + address;
                this.setSupervisionState(EC2InstanceState.RUNNING, msg, null);
                break;
            }
            case Stopping: {
                msg = "Bamboo has detected that EC2 EBS-backed instance " + this.getInstanceId() + " is stopping.";
                this.setSupervisionState(EC2InstanceState.STOPPING, msg, null);
                break;
            }
            case Stopped: {
                msg = "Bamboo has detected that EC2 EBS-backed instance " + this.getInstanceId() + " has stopped.";
                this.setSupervisionState(EC2InstanceState.STOPPED, msg, null);
                break;
            }
            case ShuttingDown: {
                msg = "Bamboo has detected that EC2 instance " + this.getInstanceId() + " is shutting down. State: " + instance.getStateDescription();
                this.instanceStatus.setDeadline(300);
                this.setSupervisionState(EC2InstanceState.SHUTTING_DOWN, msg, null);
                break;
            }
            case Terminated: {
                msg = "EC2 instance " + this.getInstanceId() + " has terminated.";
                this.setSupervisionState(EC2InstanceState.TERMINATED, msg, null);
            }
        }
        if (msg != null) {
            log.info(msg);
        }
    }

    private void updatePostLaunchData(InstanceReservationDescription instance) {
        this.instanceStatus.setInstanceId(instance.getInstanceId());
        this.instanceStatus.setAvailabilityZone(instance.getAvailabilityZone());
        this.instanceStatus.setState(instance.getState());
        InstanceType instanceType = instance.getInstanceType();
        if (instanceType != null) {
            this.instanceStatus.setInstanceType(instanceType);
        }
    }

    private synchronized void backgroundStart() {
        log.trace((Object)"Entered backgroundStart()");
        InstanceLauncher launcher = this.instanceLauncherFactory.newLauncher(this.instanceConfiguration, this.instanceStatus);
        try {
            if (this.launchInstance(launcher)) {
                this.supervisorJob = this.scheduledExecutorService.scheduleWithFixedDelay(this.supervisor, 0L, this.pollPeriodInSeconds, TimeUnit.SECONDS);
            }
        }
        finally {
            log.trace((Object)"Finished backgroundStart()");
        }
    }

    private boolean launchInstance(InstanceLauncher launcher) {
        EC2InstanceState initialState;
        String details;
        InstanceReservationDescription instance;
        this.instanceStatus.onStartupAttempt();
        try {
            Collection<InstanceReservationDescription> instances = launcher.call();
            instance = (InstanceReservationDescription)Iterables.getOnlyElement(instances);
            this.instanceStatus.setSubnetId(instance.getSubnet());
        }
        catch (Throwable throwable) {
            String details2 = "EC2 instance order for image " + this.instanceConfiguration.getImage().getId() + " failed.";
            log.error((Object)details2, throwable);
            this.setSupervisionState(EC2InstanceState.FAILED_TO_START, details2, throwable);
            return false;
        }
        if (this.instanceStatus.getInstancePaymentType() == InstancePaymentType.SPOT) {
            details = "Placed spot request " + this.instanceStatus.getSensibleId();
            initialState = EC2InstanceState.BIDDING;
        } else {
            this.instanceStatus.setInstanceId(instance.getInstanceId());
            details = "Ordered EC2 instance " + this.instanceStatus.getSensibleId();
            initialState = EC2InstanceState.PENDING;
        }
        this.instanceStatus.setInstanceType(instance.getInstanceType());
        this.instanceStatus.setAvailabilityZone(instance.getAvailabilityZone());
        log.info((Object)details);
        this.setSupervisionState(initialState, details, null);
        return true;
    }

    private synchronized void backgroundSupervise() {
        block7: {
            log.trace((Object)"Entered backgroundSupervise()");
            try {
                if (this.state.isFinal()) {
                    throw new IllegalStateException(this.instanceStatus.getSensibleId() + ": " + (Object)((Object)this.state) + " is a final state.");
                }
                try {
                    this.state.supervise(this);
                    this.successiveSupervisionFailures.set(0);
                }
                catch (Throwable throwable) {
                    if (this.successiveSupervisionFailures.incrementAndGet() > this.maxSuccessiveSupervisionFailures) {
                        log.error((Object)("The request for the current status of EC2 instance/spot request " + this.instanceStatus.getSensibleId() + " failed after " + this.maxSuccessiveSupervisionFailures + " attempts.  No further attempts will be made."), throwable);
                        this.state.supervisionFailure(this, throwable);
                        break block7;
                    }
                    Level level = this.instanceStatus.isSensibleIdOlderThan(AWS_RESOURCE_ID_PROPAGATION_TIME) ? Level.WARN : Level.DEBUG;
                    log.log((Priority)level, (Object)("The request for the current status of EC2 instance/spot request " + this.instanceStatus.getSensibleId() + " failed, try " + this.successiveSupervisionFailures.get() + ".  Will retry later."), throwable);
                }
            }
            finally {
                log.trace((Object)"Finished backgroundSupervise()");
            }
        }
    }

    void setSupervisionState(EC2InstanceState newState, @Nullable String details, @Nullable Throwable throwable) {
        if (newState == this.state) {
            return;
        }
        EC2InstanceState previousState = this.state;
        this.state = newState;
        if (newState.isFinal() && this.supervisorJob != null) {
            log.debug((Object)"Cancelling supervisor");
            this.supervisorJob.cancel(false);
        }
        CatchingRunnableDecorator catchingRunnableDecorator = new CatchingRunnableDecorator("Listener " + this.listener, () -> this.listener.ec2InstanceStateChanged(this, previousState, newState, details, throwable));
        catchingRunnableDecorator.run();
    }

    @Override
    public InstanceLaunchConfiguration getInstanceConfiguration() {
        return this.instanceConfiguration;
    }

    @Override
    public InstanceStatus getInstanceStatus() {
        return this.instanceStatus;
    }

    @Override
    public String getInstanceId() {
        return this.instanceStatus.getInstanceId();
    }

    @Override
    public void addTag(@NotNull String key, @NotNull String value) {
        CreateTagsRequest createTagsRequest = new CreateTagsRequest().withResources(new String[]{this.instanceStatus.getInstanceId()}).withTags(new Tag[]{new Tag(key, value)});
        this.awsAccount.getAmazonEc2().createTagsAsync(createTagsRequest);
    }

    @Override
    public void reconnectToRunningInstance(@NotNull Instance instance) {
        this.state = EC2InstanceState.RUNNING;
        this.startCalled.compareAndSet(false, true);
        this.instanceStatus.setInstanceType(InstanceType.fromValue((String)instance.getInstanceType()));
        this.instanceStatus.setInstanceId(instance.getInstanceId());
        this.instanceStatus.setLaunchTime(instance.getLaunchTime());
        this.instanceStatus.setState(AwsSupportConstants.InstanceStateName.fromValue(instance.getState()));
        this.instanceStatus.setStartupTime(instance.getLaunchTime().getTime());
        AwsInstanceReservationDescription instanceReservationDescription = new AwsInstanceReservationDescription(instance);
        this.instanceStatus.setAvailabilityZone(instanceReservationDescription.getAvailabilityZone());
        this.instanceStatus.setAddressAndHostname(instanceReservationDescription);
        this.instanceStatus.setSubnetId(instanceReservationDescription.getSubnet());
        if (StringUtils.isNotBlank((CharSequence)instance.getSpotInstanceRequestId())) {
            this.instanceStatus.setSpotInstanceRequestId(instance.getSpotInstanceRequestId());
            this.instanceStatus.setInstancePaymentType(InstancePaymentType.SPOT);
        } else {
            this.instanceStatus.setInstancePaymentType(InstancePaymentType.REGULAR);
        }
        this.supervisorJob = this.scheduledExecutorService.scheduleWithFixedDelay(this.supervisor, 0L, this.pollPeriodInSeconds, TimeUnit.SECONDS);
    }

    static class CatchingRunnableDecorator
    implements Runnable {
        private final String description;
        private final Runnable runnable;

        public CatchingRunnableDecorator(String description, Runnable runnable) {
            this.description = description;
            this.runnable = runnable;
        }

        @Override
        public void run() {
            try {
                log.trace((Object)("Entering " + this.description));
                this.runnable.run();
            }
            catch (Throwable throwable) {
                log.error((Object)("Exception during " + this.description), throwable);
            }
        }
    }
}

