package com.atlassian.aws.utils;

import com.google.common.collect.ComputationException;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMException;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.jetbrains.annotations.NotNull;

import javax.crypto.Cipher;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.security.KeyPair;

public final class CryptoUtils
{
    private static final JcaPEMKeyConverter JCA_PEM_KEY_CONVERTER = new JcaPEMKeyConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME);

    private CryptoUtils()
    {
    }

    @NotNull
    public static PEMKeyPair parsePemKeyPair(@NotNull final File keyPairFile) throws IOException
    {
        final FileReader br = new FileReader(keyPairFile);
        final PEMParser pemParser = new PEMParser(br);
        try
        {
            final Object object = pemParser.readObject();

            if (object instanceof PEMKeyPair)
            {
                return (PEMKeyPair)object;
            }
            else
            {
                throw new IllegalArgumentException("Don't know how to handle " + keyPairFile.getAbsolutePath() + " of type " + object.getClass().getCanonicalName());
            }
        }
        finally
        {
            pemParser.close();
        }
    }

    public static Cipher getCipherForKey(final File keyLocation, final String cipherSpec)
    {
        try
        {
            final PEMKeyPair pemKeyPair = parsePemKeyPair(keyLocation);

            final KeyPair privateKey = toJavaKeyPair(pemKeyPair);

            final Cipher cipher  = Cipher.getInstance(cipherSpec);
            cipher.init(Cipher.DECRYPT_MODE, privateKey.getPrivate());
            return cipher;
        }
        catch (final Exception e)
        {
            throw new ComputationException(e);
        }
    }

    private static KeyPair toJavaKeyPair(final PEMKeyPair keyPair) throws PEMException
    {
        return JCA_PEM_KEY_CONVERTER.getKeyPair(keyPair);
    }
}
