package com.atlassian.aws.ec2.model;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * Encapsulates a generic AWS resource string id.
 */
public abstract class ResourceId<T>
{
    private final String id;

    public ResourceId(@NotNull final String id)
    {
        this.id = Preconditions.checkNotNull(id);
    }

    @NotNull
    public final String getId()
    {
        return id;
    }


    @Override
    public final String toString()
    {
        return id;
    }

    @Override
    public final boolean equals(@Nullable final Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ResourceId<T> resourceId = (ResourceId<T>) o;

        if (!this.id.equals(resourceId.id)) return false;

        return true;
    }

    @Override
    public final int hashCode()
    {
        return id.hashCode();
    }

    @NotNull
    public static <T> Iterable<T> from(final Iterable<String> ids, final Function<String, T> function)
    {
        return StreamSupport.stream(ids.spliterator(), false)
                .map(function::apply)
                .collect(Collectors.toList());
    }

    @NotNull
    public static <T> List<T> fromCollection(final Collection<String> ids, final java.util.function.Function<String, T> function)
    {
        return ids.stream()
                .map(function)
                .collect(Collectors.toList());
    }

    @NotNull
    public static Collection<String> getIds(final Iterable<? extends ResourceId<?>> resourceIds)
    {
        return StreamSupport.stream(resourceIds.spliterator(), false)
                .map(ResourceId::getId)
                .collect(Collectors.toList());
    }
}
