package com.atlassian.aws.ec2.caches;


import com.amazonaws.services.ec2.AmazonEC2Async;
import com.amazonaws.services.ec2.model.DescribeInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeInstancesResult;
import com.amazonaws.services.ec2.model.Filter;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.Reservation;
import com.atlassian.aws.AmazonServiceErrorCode;
import com.atlassian.aws.AwsOmeCache;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

public class InstanceCache extends AwsOmeCache<Instance>
{
    private static final Logger log = Logger.getLogger(InstanceCache.class);

    private final AmazonEC2Async ec2Client;

    public InstanceCache(final AmazonEC2Async ec2Client, final Duration maximumDataAge)
    {
        super(log, maximumDataAge);
        this.ec2Client = ec2Client;
    }

    public InstanceCache(final AmazonEC2Async ec2Client)
    {
        super(log);
        this.ec2Client = ec2Client;
    }

    @NotNull
    private static Filter newDescribeInstanecsVpcGroupFilter(final String groupName)
    {
        return new Filter("instance.group-name", Collections.singletonList(groupName));
    }

    @Override
    protected List<Instance> refreshAllData()
    {
        final DescribeInstancesRequest describeInstancesRequest = new DescribeInstancesRequest();

        final String bambooControlTag = System.getProperties().getProperty("bamboo.ec2.instance.filters");
        if (bambooControlTag!=null)
        {
            describeInstancesRequest.withFilters(newDescribeInstanecsVpcGroupFilter(bambooControlTag));
        }

        final DescribeInstancesResult describeInstancesResult = ec2Client.describeInstances(describeInstancesRequest);

        return describeInstancesResult.getReservations().stream()
                .map(Reservation::getInstances)
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
    }

    @Override
    protected String toResourceId(final Instance instance)
    {
        return instance.getInstanceId();
    }

    @Override
    protected void onResourceLookupFailure(final String resourceId)
    {
        throw new Ec2CacheMissException(resourceId, AmazonServiceErrorCode.INVALID_INSTANCE_ID_NOT_FOUND);
    }
}
